#include "optimizer/hexalyoptimizer.h"
#include <algorithm>
#include <fstream>
#include <iostream>
#include <limits>
#include <numeric>
#include <vector>
#include <cstring>

using namespace hexaly;
using namespace std;

class RcpspMultiMode {
private:
    // Number of tasks
    int nbTasks;
    // Number of renewable resources
    int nbRenewableResources;
    // Total number of resources (renewable and non-renewable)
    int nbResources;
    // Number of available modes for each task
    std::vector<int> nbModes;
    // Maximum capacity of each resource
    std::vector<int> capacity;
    // Duration of each task and mode
    std::vector<std::vector<int>> duration;
    // Required resource weight for each task and mode
    std::vector<std::vector<std::vector<int>>> weight;
    // Number of successors of each task
    std::vector<int> nbSuccessors;
    // Successors of each task
    std::vector<std::vector<int>> successors;
    // Trivial upper bound for the start times of the tasks
    int horizon = 0;

    // Hexaly Optimizer
    HexalyOptimizer optimizer;
    // Decision variables: time range for each task and mode
    std::vector<std::vector<HxExpression>> tasksInMode;
    // Intermediate variables: boolean indicating for each pair (task, mode) whether it is active or not
    std::vector<std::vector<HxExpression>> presentMode;
    // Intermediate variables: time range of each task (hull on all @RcpspMultiMode#tasksInMode)
    std::vector<HxExpression> tasks;
    // Intermediate variables: number of active modes per task
    std::vector<HxExpression> nbModesPerTask;
    // Objective = minimize the makespan: end of the last task of the last job
    HxExpression makespan;

public:
    RcpspMultiMode(const std::string& fileName) : optimizer() {}

    void solve(int TimeLimit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        // Optional interval decisions: time range for each task and mode
        tasksInMode.resize(nbTasks);
        presentMode.resize(nbTasks);

        // Hull
        tasks.resize(nbTasks);

        // Declare decisions
        for (int task = 0; task < nbTasks; task++) {
            tasksInMode[task].resize(nbModes[task]);
            presentMode[task].resize(nbModes[task]);
            for (int mode = 0; mode < nbModes[task]; mode++) {
                tasksInMode[task][mode] = model.optionalIntervalVar(0, horizon);
                presentMode[task][mode] = model.presence(tasksInMode[task][mode]);
            }

            tasks[task] = model.hull(model.array(tasksInMode[task].begin(), tasksInMode[task].end()));
        }

        // Build makespan
        makespan = model.max();
        for (int task = 0; task < nbTasks; task++) {
            makespan.addOperand(model.end(tasks[task]));
        }

        // Declare constraints on tasks
        nbModesPerTask.resize(nbTasks);
        for (int task = 0; task < nbTasks; task++) {
            // Constraints: Task duration
            for (int mode = 0; mode < nbModes[task]; mode++) {
                model.constraint(model.iif(
                        presentMode[task][mode],
                        model.eq(model.length(tasksInMode[task][mode]), duration[task][mode]),
                        1
                ));
            }

            // Constraints: Precedence between tasks
            for (int successor = 0; successor < nbSuccessors[task]; successor++) {
                model.constraint(model.lt(tasks[task], tasks[successors[task][successor]]));
            }

            // Constraints: Exactly one active mode for each task
            nbModesPerTask[task] = model.sum();
            for (int mode = 0; mode < nbModes[task]; mode++) {
                nbModesPerTask[task].addOperand(presentMode[task][mode]);
            }
            model.constraint(model.eq(nbModesPerTask[task], 1));
        }

        // Constraints: Renewable resources
        for (int resource = 0; resource < nbRenewableResources; resource++) {
            HxExpression capacityRespected = model.createLambdaFunction([&](HxExpression time) {
                HxExpression totalWeight = model.sum();
                for (int task = 0; task < nbTasks; task++) {
                    for (int mode = 0; mode < nbModes[task]; mode++) {
                        totalWeight.addOperand(model.prod(
                                weight[task][mode][resource],
                                model.contains(tasksInMode[task][mode], time)));
                    }
                }
                return model.leq(totalWeight, capacity[resource]);
            });
            model.constraint(model.and_(model.range(0, makespan), capacityRespected));
        }

        // Constraints: Non-renewable resources
        for (int resource = nbRenewableResources; resource < nbResources; resource++) {
            HxExpression totalWeight = model.sum();
            for (int task = 0; task < nbTasks; task++) {
                for (int mode = 0; mode < nbModes[task]; mode++) {
                    totalWeight.addOperand(model.prod(
                            weight[task][mode][resource],
                            presentMode[task][mode]));
                }
            }
            model.constraint(model.leq(totalWeight, capacity[resource]));
        }

        // Objective: Minimize makespan
        model.minimize(makespan);

        model.close();

        // Parameterize the optimizer
        optimizer.getParam().setTimeLimit(TimeLimit);

        optimizer.solve();
    }

    /* 
    *  Write the solution in a file with the following format:
    *  - total makespan
    *  - for each task, the task ID, the active mode ID, the start and end times
    */
    void writeSolution(const std::string& fileName) {
        std::ofstream outfile(fileName.c_str());
        if (!outfile.is_open()) {
            std::cerr << "File " << fileName << " cannot be opened." << std::endl;
            exit(1);
        }
        std::cout << "Solution written in file " << fileName << std::endl;

        outfile << makespan.getValue() << std::endl;

        int activeModeId = -1;
        for (int task = 0; task < nbTasks; task++) {
            for (int mode = 0; mode < nbModes[task]; mode++) {
                if (presentMode[task][mode].getValue() == 1) {
                    activeModeId = mode;
                    break;
                }
            }
            outfile << task + 1 << " " << activeModeId + 1 << " "
                    << tasks[task].getIntervalValue().getStart() << " "
                    << tasks[task].getIntervalValue().getEnd() << std::endl;
        }
        outfile.close();
    }

    void readInstance(const std::string& fileName) {
        std::ifstream infile;
        infile.exceptions(std::ifstream::failbit | std::ifstream::badbit);
        infile.open(fileName.c_str());

        // Parse number of tasks
        string str;
        char* pch;
        char* line;
        int nbNodes;
        
        for (int i = 0; i < 6; i++) {
            getline(infile, str);
        }

        line = strdup(str.c_str());
        pch = strtok(line, ":");
        pch = strtok(NULL, ":");
        nbTasks = atoi(pch);

        // Parse number of resources
        for (int i = 0; i < 3; i++) {
            getline(infile, str);
        }

        line = strdup(str.c_str());
        pch = strtok(line, ":");
        pch = strtok(NULL, ":");
        pch = strtok(pch, " ");
        nbRenewableResources = atoi(pch);

        getline(infile, str);
        line = strdup(str.c_str());
        pch = strtok(line, ":");
        pch = strtok(NULL, ":");
        pch = strtok(pch, " ");
        int nbNonRenewableResources = atoi(pch);
        nbResources = nbRenewableResources + nbNonRenewableResources;

        // Parse successors of each task
        for (int i = 0; i < 8; i++) {
            getline(infile, str);
        }

        nbModes.resize(nbTasks);
        nbSuccessors.resize(nbTasks);
        successors.resize(nbTasks);
        
        int task;
        infile >> task;
        task--;
        while (true) {
            infile >>  nbModes[task];
            infile >>  nbSuccessors[task];
            successors[task].resize(nbSuccessors[task]);
            for (int successor = 0; successor < nbSuccessors[task]; successor++) {
                infile >> successors[task][successor];
                successors[task][successor]--;
            }
            if (task + 1 == nbTasks) {
                break;
           }
           infile >> task;
           task--;
        }

        // Parse tasks durations per mode AND consumed resource weight per mode for each task
        for (int i = 0; i < 5; i++) {
            getline(infile, str);
        }

        infile >> task;
        task--;
        duration.resize(nbTasks);
        weight.resize(nbTasks);
        int modeId;
        int maxDuration;

        while (true) {
            weight[task].resize(nbModes[task]);
            duration[task].resize(nbModes[task]);
            maxDuration = 0;
            for (int mode = 0; mode < nbModes[task]; mode++) {
                infile >> modeId;
                modeId--;
                infile >> duration[task][modeId];
                weight[task][modeId].resize(nbResources);
                maxDuration = std::max(maxDuration, duration[task][modeId]);
                for (int resourceId = 0; resourceId < nbResources; resourceId++) {
                    infile >> weight[task][modeId][resourceId];
                }
            }
            horizon += maxDuration;
            if (task + 1 == nbTasks) {
                break;
            }
            infile >> task;
            task--;
        }
        
        // Parse maximum capacity for each resource
        for (int i = 0; i < 4; i++) {
            getline(infile, str);
        }

        capacity.resize(nbResources);
        for (int resource = 0; resource < nbResources; resource++) {
            infile >> capacity[resource];
        }

        infile.close();
    }
};

int main(int argc, char** argv) {
    if (argc < 2) {
        std::cout << "Usage: RcpspMultiMode instanceFile [outputFile] [timeLimit]" << std::endl;
        exit(1);
    }

    const char* instanceFile = argv[1];
    const char* outputFile = argc > 2 ? argv[2] : NULL;
    const char* strTimeLimit = argc > 3 ? argv[3] : "60";

    RcpspMultiMode model(instanceFile);
    try {
        model.readInstance(instanceFile);
        const int timeLimit = atoi(strTimeLimit);
        model.solve(timeLimit);
        if (outputFile != NULL)
            model.writeSolution(outputFile);
        return 0;
    } catch (const std::exception& e) {
        std::cerr << "An error occurred: " << e.what() << std::endl;
        return 1;
    }
}
