import java.util.*;
import java.io.*;
import com.hexaly.optimizer.*;

public class RcpspProducerConsumer {
    // Number of tasks
    private int nbTasks;
    // Number of resources
    private int nbResources;
    // Number of inventories
    private int nbInventories;
    // Maximum capacity of each resource
    private int[] capacity;
    // Initial level of each inventory
    private int[] initLevel;
    // Duration of each task
    private int[] duration;
    // Resource weight of resource r required for task i
    private int[][] weight;
    // Inventory consumed at beginning of task
    private int[][] startCons;
    // Inventory produced at end of task
    private int[][] endProd;
    // Number of successors
    private int[] nbSuccessors;
    // Successors for each task i
    private int[][] successors;
    // Trivial upper bound for the end times of the tasks
    private int horizon = 0;

    // Hexaly Optimizer
    final HexalyOptimizer optimizer;
    // Decision variables: time range of each task
    private HxExpression[] tasks;
    // Objective = minimize the makespan: end of the last task of the last job
    private HxExpression makespan;

    public RcpspProducerConsumer(HexalyOptimizer optimizer, String fileName) throws IOException {
        this.optimizer = optimizer;
    }

    // The input files follow the "Patterson" format
    private void readInstance(String fileName) throws IOException {
        try (Scanner input = new Scanner(new File(fileName))) {
            input.useLocale(Locale.ROOT);
            nbTasks = input.nextInt();
            nbResources = input.nextInt();
            nbInventories = input.nextInt();

            // The maximum capacity of each resource
            capacity = new int[nbResources];
            for (int r = 0; r < nbResources; ++r) {
                capacity[r] = input.nextInt();
            }
            // The initial level of each inventory
            initLevel = new int[nbInventories];
            for (int r = 0; r < nbInventories; ++r) {
                initLevel[r] = input.nextInt();
            }

            duration = new int[nbTasks];
            weight = new int[nbTasks][nbResources];
            startCons = new int[nbTasks][nbInventories];
            endProd = new int[nbTasks][nbInventories];
            nbSuccessors = new int[nbTasks];
            successors = new int[nbTasks][];
            for (int i = 0; i < nbTasks; ++i) {
                duration[i] = input.nextInt();
                for (int r = 0; r < nbResources; ++r) {
                    weight[i][r] = input.nextInt();
                }
                for (int r = 0; r < nbInventories; ++r) {
                    startCons[i][r] = input.nextInt();
                    endProd[i][r] = input.nextInt();
                }
                nbSuccessors[i] = input.nextInt();
                successors[i] = new int[nbSuccessors[i]];
                for (int s = 0; s < nbSuccessors[i]; ++s) {
                    successors[i][s] = input.nextInt() - 1;
                }
                horizon += duration[i];
            }
        }
    }

    public void solve(int timeLimit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        // Interval decisions: time range of each task
        tasks = new HxExpression[nbTasks];
        for (int i = 0; i < nbTasks; ++i) {
            tasks[i] = model.intervalVar(0, horizon);

            // Task duration constraints
            model.constraint(model.eq(model.length(tasks[i]), duration[i]));
        }

        // Precedence constraints between the tasks
        for (int i = 0; i < nbTasks; ++i) {
            for (int s = 0; s < nbSuccessors[i]; ++s) {
                model.constraint(model.lt(tasks[i], tasks[successors[i][s]]));
            }
        }

        // Makespan: end of the last task
        makespan = model.max();
        for (int i = 0; i < nbTasks; ++i) {
            makespan.addOperand(model.end(tasks[i]));
        }

        // Cumulative resource constraints
        for (int r = 0; r < nbResources; ++r) {
            final int rL = r;
            HxExpression capacityRespected = model.lambdaFunction(t -> {
                HxExpression totalWeight = model.sum();
                for (int i = 0; i < nbTasks; ++i) {
                    totalWeight.addOperand(model.prod(
                            weight[i][rL],
                            model.contains(tasks[i], t)));
                }
                return model.leq(totalWeight, capacity[rL]);
            });
            model.constraint(model.and(model.range(0, makespan), capacityRespected));
        }

        // Non-negative inventories constraints
        for (int r = 0; r < nbInventories; ++r) {
            final int rL = r;
            HxExpression inventoryValue = model.lambdaFunction(t -> {
                HxExpression totalValue = model.sum();
                totalValue.addOperand(initLevel[rL]);
                for (int i = 0; i < nbTasks; ++i) {
                    totalValue.addOperand(model.sub(model.prod(
                            endProd[i][rL], 
                            model.leq(model.end(tasks[i]), t)),
                        model.prod(startCons[i][rL],
                                model.leq(model.start(tasks[i]), t))));
                }
                return model.geq(totalValue, 0);
            });
            model.constraint(model.and(model.range(0, makespan), inventoryValue));
        }

        // Minimize the makespan
        model.minimize(makespan);

        model.close();

        // Parameterize the optimizer
        optimizer.getParam().setTimeLimit(timeLimit);

        optimizer.solve();
    }

    /*
     * Write the solution in a file with the following format:
     * - total makespan
     * - for each task, the task id, the start and end times
     */
    public void writeSolution(String fileName) throws IOException {
        try (PrintWriter output = new PrintWriter(fileName)) {
            System.out.println("Solution written in file " + fileName);

            output.println(makespan.getValue());

            for (int i = 0; i < nbTasks; ++i) {
                output.println((i + 1) + " " + tasks[i].getIntervalValue().getStart() + " "
                        + tasks[i].getIntervalValue().getEnd());
            }
        }
    }

    public static void main(String[] args) {
        if (args.length < 1) {
            System.out.println("Usage: java RcpspProducerConsumer instanceFile [outputFile] [timeLimit]");
            System.exit(1);
        }

        String instanceFile = args[0];
        String outputFile = args.length > 1 ? args[1] : null;
        String strTimeLimit = args.length > 2 ? args[2] : "60";

        try (HexalyOptimizer optimizer = new HexalyOptimizer()) {
            RcpspProducerConsumer model = new RcpspProducerConsumer(optimizer, instanceFile);
            model.readInstance(instanceFile);
            model.solve(Integer.parseInt(strTimeLimit));
            if (outputFile != null) {
                model.writeSolution(outputFile);
            }
        } catch (Exception ex) {
            System.err.println(ex);
            ex.printStackTrace();
            System.exit(1);
        }
    }
}
