using System;
using System.IO;
using System.Collections.Generic;
using Hexaly.Optimizer;

public class SteelMillSlabDesign : IDisposable
{
    // Number of available slabs
    int nbSlabs;

    // Number of orders
    int nbOrders;

    // Number of colors
    int nbColors;

    // Maximum number of colors per slab
    int nbColorsMaxSlab;

    // Maximum size of a slab
    int maxSize;

    // List of colors for each order
    int[] colorsData;

    // List of quantities for each order
    int[] quantitiesData;

    // Steel waste computed for each content value
    long[] wasteForContent;

    // Hexaly Optimizer
    HexalyOptimizer optimizer;

    // Hexaly Program variables
    HxExpression[] slabs;

    // Objective
    HxExpression totalWastedSteel;

    public SteelMillSlabDesign()
    {
        optimizer = new HexalyOptimizer();
    }

    public void Dispose()
    {
        if (optimizer != null)
            optimizer.Dispose();
    }

    /* Read instance data */
    void ReadInstance(string fileName)
    {
        using (StreamReader input = new StreamReader(fileName))
        {
            nbColorsMaxSlab = 2;
            string[] splitted = input.ReadLine().Split();
            int nbSlabSizes = int.Parse(splitted[0]);
            int[] slabSizes = new int[nbSlabSizes];
            for (int i = 0; i < nbSlabSizes; ++i)
                slabSizes[i] = int.Parse(splitted[i + 1]);
            maxSize = slabSizes[nbSlabSizes - 1];

            nbColors = int.Parse(input.ReadLine());
            nbOrders = int.Parse(input.ReadLine());
            nbSlabs = nbOrders;

            int sumSizeOrders = 0;
            
            quantitiesData = new int[nbOrders];
            colorsData = new int[nbOrders];
            for (int o = 0; o < nbOrders; ++o)
            {
                splitted = input.ReadLine().Split();
                quantitiesData[o] = int.Parse(splitted[0]);
                int c = int.Parse(splitted[1]);
                // Note: colors are in 1..nbColors
                colorsData[o] = c;
                sumSizeOrders += quantitiesData[o];
            }
            PreComputeWasteForContent(slabSizes, sumSizeOrders);
        }
    }

    // Compute the vector wasteForContent
    private void PreComputeWasteForContent(int[] slabSizes, int sumSizeOrders)
    {
        // No waste when a slab is empty
        wasteForContent = new long[sumSizeOrders];

        int prevSize = 0;
        for (int i = 0; i < slabSizes.Length; ++i)
        {
            int size = slabSizes[i];
            if (size < prevSize)
                throw new Exception("Slab sizes should be sorted in ascending order");

            for (int content = prevSize + 1; content < size; ++content)
                wasteForContent[content] = size - content;
            prevSize = size;
        }
    }

    void Solve(int limit)
    {
        // Declare the optimization model
        HxModel model = optimizer.GetModel();

        // Create a HexalyOptimizer array and a function to retrieve the orders's colors and quantities
        HxExpression colors = model.Array(colorsData);
        HxExpression colorsLambda = model.LambdaFunction(i => colors[i]);
        HxExpression quantities = model.Array(quantitiesData);
        HxExpression quantitiesLambda = model.LambdaFunction(i => quantities[i]);

        HxExpression[] slabContent = new HxExpression[nbSlabs];
        HxExpression[] wastedSteel = new HxExpression[nbSlabs];
        // Create a HexalyOptimizer array to be able to access it with "at" operators
        HxExpression wasteForContentArray = model.Array(wasteForContent);

        // Set decisions: slabs[k] represents the orders in slab k
        slabs = new HxExpression[nbSlabs];
        for (int s = 0; s < nbSlabs; ++s)
        {
            slabs[s] = model.Set(nbOrders);
        }

        // Each order must be in one slab and one slab only
        model.Constraint(model.Partition(slabs));

        for (int s = 0; s < nbSlabs; ++s)
        {
            // The number of colors per slab must not exceed a specified value
            model.Constraint(model.Count(model.Distinct(slabs[s], colorsLambda)) <=nbColorsMaxSlab);

            // The content of each slab must not exceed the maximum size of the slab
            slabContent[s] = model.Sum(slabs[s], quantitiesLambda);
            model.Constraint(slabContent[s] <= maxSize);

            // Wasted steel is computed according to the content of the slab
            wastedSteel[s] = wasteForContentArray[slabContent[s]];
        }

        // Minimize the total wasted steel
        totalWastedSteel = model.Sum(wastedSteel);
        model.Minimize(totalWastedSteel);

        model.Close();

        // Parametrize the optimizer
        optimizer.GetParam().SetTimeLimit(limit);
        optimizer.Solve();
    }

    /* Write the solution in a file with the following format:
     *  - total wasted steel
     *  - number of slabs used
     *  - for each slab used, the number of orders in the slab and the list of orders */
    void WriteSolution(string fileName)
    {
        using (StreamWriter output = new StreamWriter(fileName))
        {
            output.WriteLine(totalWastedSteel.GetValue());
            int actualNbSlabs = 0;
            for (int s = 0; s < nbSlabs; ++s)
            {
                if (slabs[s].GetCollectionValue().Count() > 0)
                {
                    actualNbSlabs++;
                }
            }
            output.WriteLine(actualNbSlabs);

            for (int s = 0; s < nbSlabs; ++s)
            {
                HxCollection slabCollection = slabs[s].GetCollectionValue();
                int nbOrdersInSlab = slabCollection.Count();
                if (nbOrdersInSlab == 0) continue;
                output.Write(nbOrdersInSlab);
                for (int o = 0; o < nbOrdersInSlab; ++o)
                {
                    output.Write(" " + (slabCollection.Get(o) + 1));
                }
                output.WriteLine();
            }
        }
    }

    public static void Main(string[] args)
    {
        if (args.Length < 1)
        {
            Console.WriteLine("Usage: SteelMillSlabDesign inputFile [outputFile] [timeLimit]");
            Environment.Exit(1);
        }

        string instanceFile = args[0];
        string outputFile = args.Length > 1 ? args[1] : null;
        string strTimeLimit = args.Length > 2 ? args[2] : "60";

        using (SteelMillSlabDesign model = new SteelMillSlabDesign())
        {
            model.ReadInstance(instanceFile);
            model.Solve(int.Parse(strTimeLimit));
            if (outputFile != null)
                model.WriteSolution(outputFile);
        }
    }
}
