import java.util.*;
import java.io.*;
import com.hexaly.optimizer.*;

public class SteelMillSlabDesign {
    // Number of available slabs
    private int nbSlabs;

    // Number of orders
    private int nbOrders;

    // Number of colors
    private int nbColors;

    // Maximum number of colors per slab
    private int nbColorsMaxSlab;

    // Maximum size of a slab
    private int maxSize;

    // List of colors for each order
    private int[] colorsData;

    // List of quantities for each order
    private int[] quantitiesData;

    // Steel waste computed for each content value
    private long[] wasteForContentData;

    // Hexaly Optimizer
    private final HexalyOptimizer optimizer;

    // Objective
    private HxExpression totalWastedSteel;

    // Hexaly Program variables
    private HxExpression[] slabs;

    private SteelMillSlabDesign(HexalyOptimizer optimizer) {
        this.optimizer = optimizer;
    }

    /* Read instance data */
    private void readInstance(String fileName) throws IOException {
        try (Scanner input = new Scanner(new File(fileName))) {
            input.useLocale(Locale.ROOT);
            nbColorsMaxSlab = 2;
            int nbSlabSizes = input.nextInt();

            int[] slabSizes = new int[nbSlabSizes];
            for (int i = 0; i < nbSlabSizes; ++i) {
                slabSizes[i] = input.nextInt();
            }
            maxSize = slabSizes[nbSlabSizes - 1];

            nbColors = input.nextInt();
            nbOrders = input.nextInt();
            nbSlabs = nbOrders;

            int sumSizeOrders = 0;

            colorsData = new int[nbOrders];
            quantitiesData = new int[nbOrders];
            for (int o = 0; o < nbOrders; ++o) {
                quantitiesData[o] = input.nextInt();
                int c = input.nextInt();
                // Note: colors are in 1..nbColors
                colorsData[o] = c;
                sumSizeOrders += quantitiesData[o];
            }

            preComputeWasteForContent(slabSizes, sumSizeOrders);
        }
    }

    // Compute the vector wasteForContent
    private void preComputeWasteForContent(int[] slabSizes, int sumSizeOrders) {
        // No waste when a slab is empty
        wasteForContentData = new long[sumSizeOrders];

        int prevSize = 0;
        for (int i = 0; i < slabSizes.length; ++i) {
            int size = slabSizes[i];
            if (size < prevSize)
                throw new RuntimeException("Slab sizes should be sorted in ascending order");
            for (int content = prevSize + 1; content < size; ++content) {
                wasteForContentData[content] = size - content;
            }
            prevSize = size;
        }
    }

    private void solve(int limit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        // Create a HexalyOptimizer array and a function to retrieve the orders's colors and quantities
        HxExpression colors = model.array(colorsData);
        HxExpression colorLambda = model.lambdaFunction(i -> model.at(colors, i));
        HxExpression quantities = model.array(quantitiesData);
        HxExpression quantitiesLambda = model.lambdaFunction(i -> model.at(quantities, i));
        
        HxExpression[] slabContent = new HxExpression[nbSlabs];
        HxExpression[] wastedSteel = new HxExpression[nbSlabs];
        // Create a HexalyOptimizer array to be able to access it with "at" operators
        HxExpression wasteForContent = model.array(wasteForContentData);

        // Set decisions: slabs[k] represents the orders in slab k
        slabs = new HxExpression[nbSlabs];
        for (int s = 0; s < nbSlabs; ++s) {
            slabs[s] = model.setVar(nbOrders);
        }

        // Each order must be in one slab and one slab only
        model.constraint(model.partition(slabs));

        for (int s = 0; s < nbSlabs; ++s)  {

            HxExpression orders = slabs[s];

            // The number of colors per slab must not exceed a specified value
            model.constraint(model.leq(model.count(model.distinct(orders, colorLambda)), nbColorsMaxSlab));

            // The content of each slab must not exceed the maximum size of the slab
            slabContent[s] = model.sum(orders, quantitiesLambda);
            model.constraint(model.leq(slabContent[s], maxSize));

            // Wasted steel is computed according to the content of the slab
            wastedSteel[s] = model.at(wasteForContent, slabContent[s]);
        }

        // Minimize the total wasted steel
        totalWastedSteel = model.sum(wastedSteel);
        model.minimize(totalWastedSteel);

        model.close();

        // Parametrize the optimizer
        optimizer.getParam().setTimeLimit(limit);
        optimizer.solve();
    }

    /*
     * Write the solution in a file with the following format:
     * - total wasted steel
     * - number of slabs used
     * - for each slab used, the number of orders in the slab and the list of orders
     */
    private void writeSolution(String fileName) throws IOException {
        try (PrintWriter output = new PrintWriter(new FileWriter(fileName))) {
            output.println(totalWastedSteel.getValue());

            int actualNbSlabs = 0;
            for(int s = 0; s < nbSlabs; ++s) {
                if (slabs[s].getCollectionValue().count() > 0) {
                    actualNbSlabs++;
                }
            }
            output.println(actualNbSlabs);

            for (int s = 0; s < nbSlabs; ++s) {
                int nbOrdersInSlab = slabs[s].getCollectionValue().count();
                if (nbOrdersInSlab == 0) continue;
                output.print(nbOrdersInSlab);

                for(int o = 0;  o < nbOrdersInSlab; ++o) {
                    output.print(" " + (slabs[s].getCollectionValue().get(o) + 1));
                }
                output.println();
            }
        }
    }

    public static void main(String[] args) {
        if (args.length < 1) {
            System.err.println("Usage: java SteelMillSlabDesign inputFile [outputFile] [timeLimit]");
            System.exit(1);
        }

        String instanceFile = args[0];
        String outputFile = args.length > 1 ? args[1] : null;
        String strTimeLimit = args.length > 2 ? args[2] : "60";

        try (HexalyOptimizer optimizer = new HexalyOptimizer()) {
            SteelMillSlabDesign model = new SteelMillSlabDesign(optimizer);
            model.readInstance(instanceFile);
            model.solve(Integer.parseInt(strTimeLimit));
            if (outputFile != null) {
                model.writeSolution(outputFile);
            }
        } catch (Exception ex) {
            System.err.println(ex);
            ex.printStackTrace();
            System.exit(1);
        }
    }
}
