#include "optimizer/hexalyoptimizer.h"
#include <fstream>
#include <iostream>
#include <vector>

using namespace hexaly;
using namespace std;

class SteelMillSlabDesign {
public:
    // Number of available slabs
    int nbSlabs;
    
    // Number of orders
    int nbOrders;

    // Number of colors
    int nbColors;

    // Maximum number of colors per slab
    int nbColorsMaxSlab;

    // Maximum size of a slab
    int maxSize;

    // List of colors for each order
    vector<int> colors;

    // List of quantities for each order
    vector<int> quantities;    

    // Steel waste computed for each content value
    vector<int> wasteForContent;

    // Hexaly Optimizer
    HexalyOptimizer optimizer;

    // Hexaly Program variables
    vector<HxExpression> slabs;

    // Objective
    HxExpression totalWastedSteel;

    /* Read instance data */
    void readInstance(const string& fileName) {
        ifstream infile;
        infile.exceptions(ifstream::failbit | ifstream::badbit);
        infile.open(fileName.c_str());

        nbColorsMaxSlab = 2;

        int nbSlabSizes;
        infile >> nbSlabSizes;

        vector<int> slabSizes(nbSlabSizes);
        for (int i = 0; i < nbSlabSizes; ++i) {
            infile >> slabSizes[i];
        }
        maxSize = slabSizes[nbSlabSizes - 1];

        infile >> nbColors;
        infile >> nbOrders;
        nbSlabs = nbOrders;

        quantities.resize(nbOrders);
        colors.resize(nbOrders);
        int sumSizeOrders = 0;
        for (int o = 0; o < nbOrders; ++o) {
            infile >> quantities[o];
            // Note: colors are in 1..nbColors
            infile >> colors[o];
            sumSizeOrders += quantities[o];
        }

        preComputeWasteForContent(slabSizes, sumSizeOrders);
    }

private:
    // Compute the vector wasteForContent
    void preComputeWasteForContent(const vector<int>& slabSizes, int sumSizeOrders) {

        // No waste when a slab is empty
        wasteForContent.resize(sumSizeOrders, (int)0);

        int prevSize = 0;
        for (size_t i = 0; i < slabSizes.size(); ++i) {
            int size = slabSizes[i];
            if (size < prevSize) {
                cerr << "Slab sizes should be sorted in ascending order" << endl;
                exit(1);
            }
            for (int content = prevSize + 1; content < size; ++content) {
                wasteForContent[content] = (int)(size - content);
            }
            prevSize = size;
        }
    }

public:
    void solve(int limit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        // Create a HexalyOptimizer array and a function to retrieve the orders's colors and quantities
        HxExpression colorsArray = model.array(colors.begin(), colors.end());
        HxExpression colorLambda = model.createLambdaFunction([&](HxExpression i) { return colorsArray[i]; });
        HxExpression quantitiesArray = model.array(quantities.begin(), quantities.end());
        HxExpression quantitiesLambda = model.createLambdaFunction([&](HxExpression i) { return quantitiesArray[i]; });

        vector<HxExpression> slabContent(nbSlabs);
        vector<HxExpression> wastedSteel(nbSlabs);
        // Create a HexalyOptimizer array to be able to access it with "at" operators
        HxExpression wasteForContentArray = model.array(wasteForContent.begin(), wasteForContent.end());

        // Set decisions: slab[k] represents the orders in slab k
        slabs.resize(nbSlabs);
        for (int s = 0; s < nbSlabs; ++s)  {
            slabs[s] = model.setVar(nbOrders);
        }

        // Each order must be in one slab and one slab only
        model.constraint(model.partition(slabs.begin(), slabs.end()));

        for (int s = 0; s < nbSlabs; ++s) {

            HxExpression orders = slabs[s];

            // The number of colors per slab must not exceed a specified value
            model.constraint(model.count(model.distinct(orders, colorLambda)) <= nbColorsMaxSlab);

            // The content of each slab must not exceed the maximum size of the slab
            slabContent[s] = model.sum(orders, quantitiesLambda);
            model.constraint(slabContent[s] <= maxSize);

            // Wasted steel is computed according to the content of the slab
            wastedSteel[s] = wasteForContentArray[slabContent[s]];
        }

    // Minimize the total wasted steel
    totalWastedSteel = model.sum(wastedSteel.begin(), wastedSteel.end());
    model.minimize(totalWastedSteel);

    model.close();

    // Parametrize the optimizer
    optimizer.getParam().setTimeLimit(limit);
    optimizer.solve();
    }

    /* Write the solution in a file with the following format:
     *  - total wasted steel
     *  - number of slabs used
     *  - for each slab used, the number of orders in the slab and the list of orders */
    void writeSolution(const string& fileName) {
        ofstream outfile;
        outfile.exceptions(ofstream::failbit | ofstream::badbit);
        outfile.open(fileName.c_str());
        outfile << totalWastedSteel.getValue() << endl;

        int actualNbSlabs = 0;
        for(int s = 0; s < nbSlabs; ++s) {
            if (slabs[s].getCollectionValue().count() > 0) actualNbSlabs++;
        }
        outfile << actualNbSlabs << endl;

        for (int s = 0; s < nbSlabs; ++s) {
            HxCollection slabCollection = slabs[s].getCollectionValue();
            int nbOrdersInSlab = slabCollection.count();
            if (nbOrdersInSlab == 0) continue;
            outfile << nbOrdersInSlab;
            for(int o = 0; o < nbOrdersInSlab; ++o) {
                outfile << " " << slabCollection.get(o) + 1;
            }
            outfile << endl;
        }
    }
};

int main(int argc, char** argv) {
    if (argc < 2) {
        cerr << "Usage: steel_mill_slab_design inputFile [outputFile] [timeLimit]" << endl;
        return 1;
    }

    const char* instanceFile = argv[1];
    const char* outputFile = argc >= 3 ? argv[2] : NULL;
    const char* strTimeLimit = argc >= 4 ? argv[3] : "60";

    try {
        SteelMillSlabDesign model;
        model.readInstance(instanceFile);
        model.solve(atoi(strTimeLimit));
        if (outputFile != NULL)
            model.writeSolution(outputFile);
        return 0;
    } catch (const exception& e) {
        cerr << "An error occurred: " << e.what() << endl;
        return 1;
    }
}
