import hexaly.optimizer
import sys

if len(sys.argv) < 2:
    print("Usage: python steel_mill_slab_design.py inputFile [outputFile] [timeLimit]")
    sys.exit(1)

def read_integers(filename):
    with open(filename) as f:
        return [int(elem) for elem in f.read().split()]


# Compute the vector waste_for_content
def pre_compute_waste_for_content(slab_sizes, sum_size_orders):
    # No waste when a slab is empty
    waste_for_content = [0] * sum_size_orders

    prev_size = 0
    for size in slab_sizes:
        if size < prev_size:
            print("Slab sizes should be sorted in ascending order")
            sys.exit(1)
        for content in range(prev_size + 1, size):
            waste_for_content[content] = size - content
        prev_size = size
    return waste_for_content


with hexaly.optimizer.HexalyOptimizer() as optimizer:
    #
    # Read instance data
    #
    nb_colors_max_slab = 2

    file_it = iter(read_integers(sys.argv[1]))
    nb_slab_sizes = next(file_it)
    slab_sizes = [next(file_it) for i in range(nb_slab_sizes)]
    max_size = slab_sizes[nb_slab_sizes - 1]

    nb_colors = next(file_it)
    nb_orders = next(file_it)
    nb_slabs = nb_orders

    sum_size_orders = 0

    # List of quantities and colors for each order
    quantities_data = []
    colors_data = []
    for o in range(nb_orders):
        quantities_data.append(next(file_it))
        colors_data.append(next(file_it))
        sum_size_orders += quantities_data[o]

    waste_for_content = pre_compute_waste_for_content(slab_sizes, sum_size_orders)

    #
    # Declare the optimization model
    #
    model = optimizer.model

    # Create array and function to retrieve the orders's colors and quantities
    colors = model.array(colors_data)
    color_lambda = model.lambda_function(lambda l: colors[l])
    quantities = model.array(quantities_data)
    quantity_lambda = model.lambda_function(lambda o: quantities[o])

    # Set decisions: slab[k] represents the orders in slab k
    slabs = [model.set(nb_orders) for s in range(nb_slabs)]

    # Each order must be in one slab and one slab only
    model.constraint(model.partition(slabs))

    slabContent = []
    
    for s in range(nb_slabs):

        # The number of colors per slab must not exceed a specified value
        model.constraint(model.count(model.distinct(slabs[s], color_lambda)) <= nb_colors_max_slab)

        # The content of each slab must not exceed the maximum size of the slab
        slabContent.append(model.sum(slabs[s], quantity_lambda))
        model.constraint(slabContent[s] <= max_size)

    waste_for_content_array = model.array(waste_for_content)

    # Wasted steel is computed according to the content of the slab
    wasted_steel = [waste_for_content_array[slabContent[s]] for s in range(nb_slabs)]

    # Minimize the total wasted steel
    total_wasted_steel = model.sum(wasted_steel)
    model.minimize(total_wasted_steel)

    model.close()

    # Parameterize the optimizer
    if len(sys.argv) >= 4:
        optimizer.param.time_limit = int(sys.argv[3])
    else:
        optimizer.param.time_limit = 60
    optimizer.solve()

    #
    # Write the solution in a file with the following format:
    #  - total wasted steel
    #  - number of slabs used
    #  - for each slab used, the number of orders in the slab and the list of orders
    #
    if len(sys.argv) >= 3:
        with open(sys.argv[2], 'w') as f:
            f.write("%d\n" % total_wasted_steel.value)
            actual_nb_slabs = 0
            for s in range(nb_slabs):
                if slabs[s].value.count() > 0:
                    actual_nb_slabs += 1
            f.write("%d\n" % actual_nb_slabs)

            for s in range(nb_slabs):
                nb_orders_in_slab = slabs[s].value.count()
                if nb_orders_in_slab == 0:
                    continue
                f.write("%d" % nb_orders_in_slab)
                for o in slabs[s].value:
                    f.write(" %d" % (o + 1))
                f.write("\n")
