import java.util.*;
import java.io.*;
import com.hexaly.optimizer.*;

public class StochasticJobshop {
    // Number of jobs
    private int nbJobs;
    // Number of machines
    private int nbMachines;
    // Number of scenarios
    private int nbScenarios;
    // Processing time on each machine for each job (given in the machine order)
    private long[][][] processingTimePerScenario;
    // Processing order of machines for each job
    private int[][] machineOrder;
    // Trivial upper bound for the end times of the tasks
    private long maxEnd;

    // Hexaly Optimizer
    final HexalyOptimizer optimizer;
    // Decision variables: time range of each task
    private HxExpression[][][] tasks;
    // Decision variables: sequence of tasks on each machine
    private HxExpression[] jobsOrder;
    // Objective = minimize the maximum of all makespans
    private HxExpression maxMakespan;

    public StochasticJobshop(HexalyOptimizer optimizer) throws IOException {
        this.optimizer = optimizer;
    }

    public void readInstance(String fileName) throws IOException {
        try (Scanner input = new Scanner(new File(fileName))) {
            input.useLocale(Locale.ROOT);
            input.nextLine();
            nbJobs = input.nextInt();
            nbMachines = input.nextInt();
            nbScenarios = input.nextInt();

            input.nextLine();
            input.nextLine();
            // Processing times for each job on each machine (given in the processing order)
            long[][][] processingTimesInProcessingOrderPerScenario = new long[nbScenarios][nbJobs][nbMachines];
            for (int s = 0; s < nbScenarios; ++s) {
                for (int j = 0; j < nbJobs; ++j) {
                    for (int m = 0; m < nbMachines; ++m) {
                        processingTimesInProcessingOrderPerScenario[s][j][m] = input.nextInt();
                    }
                }
                input.nextLine();
            }
            // Processing order of machines for each job
            input.nextLine();
            input.nextLine();
            machineOrder = new int[nbJobs][nbMachines];
            for (int j = 0; j < nbJobs; ++j) {
                for (int m = 0; m < nbMachines; ++m) {
                    machineOrder[j][m] = input.nextInt() - 1;
                }
            }

            // Reorder processing times: processingTimePerScenario[s][j][m] is the
            // processing time of the task of job j that is processed on machine m for a
            // given scenario s
            processingTimePerScenario = new long[nbScenarios][nbJobs][nbMachines];

            // Trivial upper bound for the end times of the tasks
            long[] maxEndPerScenario = new long[nbScenarios];
            for (int s = 0; s < nbScenarios; ++s) {
                maxEndPerScenario[s] = 0;
                for (int j = 0; j < nbJobs; ++j) {
                    for (int m = 0; m < nbMachines; ++m) {
                        int machineIndex = nbMachines;
                        for (int k = 0; k < nbMachines; ++k) {
                            if (machineOrder[j][k] == m) {
                                machineIndex = k;
                                break;
                            }
                        }
                        processingTimePerScenario[s][j][m] = processingTimesInProcessingOrderPerScenario[s][j][machineIndex];
                        maxEndPerScenario[s] += processingTimePerScenario[s][j][m];
                    }
                }
            }
            maxEnd = Arrays.stream(maxEndPerScenario).max().getAsLong();
        }
    }

    public void solve(int timeLimit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        // Interval decisions: time range of each task tasks[s][j][m] is the interval of
        // time of the task of job j which is processed on machine m in the scenario s
        tasks = new HxExpression[nbScenarios][nbJobs][nbMachines];
        for (int j = 0; j < nbJobs; ++j) {
            for (int m = 0; m < nbMachines; ++m) {
                for (int s = 0; s < nbScenarios; ++s) {
                    tasks[s][j][m] = model.intervalVar(0, maxEnd);
                    // Task duration constraints
                    model.constraint(model.eq(model.length(tasks[s][j][m]), processingTimePerScenario[s][j][m]));
                }
            }
        }

        // Create a HexalyOptimizer array in order to be able to access it with "at"
        // operators
        HxExpression taskArray = model.array(tasks);

        // Precedence constraints between the tasks of a job
        for (int s = 0; s < nbScenarios; ++s) {
            for (int j = 0; j < nbJobs; ++j) {
                for (int k = 0; k < nbMachines - 1; ++k) {
                    model.constraint(model.lt(tasks[s][j][machineOrder[j][k]], tasks[s][j][machineOrder[j][k + 1]]));
                }
            }
        }

        // Sequence of tasks on each machine
        jobsOrder = new HxExpression[nbMachines];
        for (int m = 0; m < nbMachines; ++m) {
            jobsOrder[m] = model.listVar(nbJobs);
        }

        for (int m = 0; m < nbMachines; ++m) {
            // Each job has a task scheduled on each machine
            HxExpression sequence = jobsOrder[m];
            model.constraint(model.eq(model.count(sequence), nbJobs));

            // Disjunctive resource constraints between the tasks on a machine
            for (int s = 0; s < nbScenarios; ++s) {
                HxExpression mExpr = model.createConstant(m);
                HxExpression sExpr = model.createConstant(s);
                HxExpression sequenceLambda = model
                        .lambdaFunction(i -> model.lt(model.at(taskArray, sExpr, model.at(sequence, i), mExpr),
                                model.at(taskArray, sExpr, model.at(sequence, model.sum(i, 1)), mExpr)));
                model.constraint(model.and(model.range(0, nbJobs - 1), sequenceLambda));
            }
        }

        // Minimize the maximum makespan: end of the last task of the last job
        // over all scenarios
        HxExpression[] makespans = new HxExpression[nbScenarios];
        for (int s = 0; s < nbScenarios; ++s) {
            makespans[s] = model.max();
            for (int j = 0; j < nbJobs; ++j) {
                makespans[s].addOperand(model.end(tasks[s][j][machineOrder[j][nbMachines - 1]]));
            }
        }

        maxMakespan = model.max();
        for (int s = 0; s < nbScenarios; ++s) {
            maxMakespan.addOperand(makespans[s]);
        }

        model.minimize(maxMakespan);

        model.close();

        // Parameterize the optimizer
        optimizer.getParam().setTimeLimit(timeLimit);

        optimizer.solve();
    }

    /*
     * Write the solution in a file with the following format:
     * - for each machine, the job sequence
     */
    public void writeSolution(String fileName) throws IOException {
        try (PrintWriter output = new PrintWriter(fileName)) {
            System.out.println("Solution written in file " + fileName);

            for (int m = 0; m < nbMachines; ++m) {
                HxCollection finalJobsOrder = jobsOrder[m].getCollectionValue();
                for (int i = 0; i < nbJobs; ++i) {
                    int j = Math.toIntExact(finalJobsOrder.get(i));
                    output.write(j + " ");
                }
                output.write("\n");
            }
        }
    }

    public static void main(String[] args) {
        if (args.length < 1) {
            System.out.println("Usage: java StochasticJobshop instanceFile [outputFile] [timeLimit]");
            System.exit(1);
        }

        String instanceFile = args[0];
        String outputFile = args.length > 1 ? args[1] : null;
        String strTimeLimit = args.length > 2 ? args[2] : "60";

        try (HexalyOptimizer optimizer = new HexalyOptimizer()) {
            StochasticJobshop model = new StochasticJobshop(optimizer);
            model.readInstance(instanceFile);
            model.solve(Integer.parseInt(strTimeLimit));
            if (outputFile != null) {
                model.writeSolution(outputFile);
            }
        } catch (Exception ex) {
            System.err.println(ex);
            ex.printStackTrace();
            System.exit(1);
        }
    }
}
