using System;

using Hexaly.Optimizer;

public class StochasticPacking : IDisposable
{
    // Number of items
    int nbItems;

    // Number of bins
    int nbBins;

    // Number of scenarios
    int nbScenarios;

    // For each scenario, the weight of each item
    int[][] scenarioItemWeights;

    // Hexaly Optimizer
    HexalyOptimizer optimizer;

    // Decision variable for the assignment of items
    HxExpression[] bins;

    // For each scenario, the corresponding maximum weight
    HxExpression[] scenarioMaxWeight;

    // Objective = minimize the 9th decile of all possible max weights
    HxExpression stochasticMaxWeight;

    private void generateScenarios(int rngSeed)
    {
        Random rng = new Random(rngSeed);

        // Pick random parameters for each item distribution
        int[] itemsMin = new int[nbItems];
        int[] itemsMax = new int[nbItems];
        for (int i = 0; i < nbItems; ++i)
        {
            itemsMin[i] = rng.Next(10, 101);
            itemsMax[i] = itemsMin[i] + rng.Next(51);
        }

        // Sample the distributions to generate the scenarios
        scenarioItemWeights = new int[nbScenarios][];
        for (int i = 0; i < nbScenarios; ++i)
        {
            scenarioItemWeights[i] = new int[nbItems];
            for (int j = 0; j < nbItems; ++j)
                scenarioItemWeights[i][j] = rng.Next(itemsMin[i], itemsMax[i] + 1);
        }
    }

    public StochasticPacking(int nbItems, int nbBins, int nbScenarios, int rngSeed)
    {
        optimizer = new HexalyOptimizer();
        this.nbItems = nbItems;
        this.nbBins = nbBins;
        this.nbScenarios = nbScenarios;
        generateScenarios(rngSeed);
    }

    public void Dispose()
    {
        if (optimizer != null)
            optimizer.Dispose();
    }

    void Solve(int limit)
    {
        // Declare the optimization model
        HxModel model = optimizer.GetModel();

        bins = new HxExpression[nbBins];
        scenarioMaxWeight = new HxExpression[nbScenarios];

        // Set decisions: bins[k] represents the items in bin k
        for (int k = 0; k < nbBins; ++k)
            bins[k] = model.Set(nbItems);

        // Each item must be in one bin and one bin only
        model.Constraint(model.Partition(bins));

        // Compute max weight for each scenario
        for (int m = 0; m < nbScenarios; ++m)
        {
            HxExpression scenario = model.Array(scenarioItemWeights[m]);
            HxExpression weightLambda = model.LambdaFunction(i => scenario[i]);
            HxExpression[] binWeights = new HxExpression[nbBins];

            for (int k = 0; k < nbBins; ++k)
                binWeights[k] = model.Sum(bins[k], weightLambda);
            scenarioMaxWeight[m] = model.Max(binWeights);
        }

        // Compute the 9th decile of scenario max weights
        HxExpression scenarioMaxWeightArray = model.Array(scenarioMaxWeight);
        HxExpression sortedScenarioMaxWeight = model.Sort(scenarioMaxWeightArray);
        stochasticMaxWeight = sortedScenarioMaxWeight[(int)Math.Ceiling(0.9 * (nbScenarios - 1))];

        model.Minimize(stochasticMaxWeight);
        model.Close();

        // Parametrize the optimizer
        optimizer.GetParam().SetTimeLimit(limit);

        optimizer.Solve();
    }

    /* Write the solution */
    private void WriteSolution()
    {
        Console.WriteLine();
        Console.WriteLine("Scenario item weights:");
        for (int i = 0; i < nbScenarios; ++i)
        {
            Console.Write(i + ": [");
            for (int j = 0; j < nbItems; ++j)
                Console.Write(scenarioItemWeights[i][j] + (j == nbItems - 1 ? "" : ", "));
            Console.WriteLine("]");
        }

        Console.WriteLine();
        Console.WriteLine("Bins:");

        for (int m = 0; m < nbBins; ++m)
        {
            Console.Write(m + ": { ");
            HxCollection items = bins[m].GetCollectionValue();
            for (int i = 0; i < items.Count(); ++i)
                Console.Write(items.Get(i) + (i == items.Count() - 1 ? " " : ", "));
            Console.WriteLine("}");
        }
    }

    public static void Main(string[] args)
    {
        int nbItems = 10;
        int nbBins = 2;
        int nbScenarios = 3;
        int rngSeed = 43;
        int timeLimit = 2;

        using (
            StochasticPacking model = new StochasticPacking(
                nbItems,
                nbBins,
                nbScenarios,
                rngSeed
            )
        )
        {
            model.Solve(timeLimit);
            model.WriteSolution();
        }
    }
}
