from __future__ import print_function
import random
import math
import hexaly.optimizer


def generate_scenarios(nb_items, nb_scenarios, rng_seed):
    random.seed(rng_seed)

    # Pick random parameters for each item distribution
    items_dist = []
    for _ in range(nb_items):
        item_min = random.randint(10, 100)
        item_max = item_min + random.randint(0, 50)
        items_dist.append((item_min, item_max))

    # Sample the distributions to generate the scenarios
    scenario_item_weights = [[random.randint(*dist) for dist in items_dist]
                               for _ in range(nb_scenarios)]
    return scenario_item_weights


def main(nb_items, nb_bins, nb_scenarios, seed, time_limit):
    # Generate instance data
    scenario_item_weights_data = generate_scenarios(nb_items, nb_scenarios, seed)

    with hexaly.optimizer.HexalyOptimizer() as optimizer:
        #
        # Declare the optimization model
        #
        model = optimizer.model

        # Set decisions: bins[k] represents the items in bin k
        bins = [model.set(nb_items) for _ in range(nb_bins)]

        # Each item must be in one bin and one bin only
        model.constraint(model.partition(bins))

        scenarios_item_weights = model.array(scenario_item_weights_data)

        # Compute max weight for each scenario
        scenarios_max_weights = model.array(
            model.max(
                model.sum(bin,
                          model.lambda_function(
                              lambda i:
                                  model.at(scenarios_item_weights, k, i)))
                for bin in bins) for k in range(nb_scenarios))

        # Compute the 9th decile of scenario max weights
        stochastic_max_weight = \
            model.sort(scenarios_max_weights)[int(math.ceil(0.9 * (nb_scenarios - 1)))]

        model.minimize(stochastic_max_weight)
        model.close()

        # Parameterize the optimizer
        optimizer.param.time_limit = time_limit

        optimizer.solve()

        #
        # Write the solution
        #
        print()
        print("Scenario item weights:")
        for i, scenario in enumerate(scenario_item_weights_data):
            print(i, ': ', scenario, sep='')

        print()
        print("Bins:")
        for k, bin in enumerate(bins):
            print(k, ': ', bin.value, sep='')


if __name__ == '__main__':
    nb_items = 10
    nb_bins = 2
    nb_scenarios = 3
    rng_seed = 42
    time_limit = 2

    main(
        nb_items,
        nb_bins,
        nb_scenarios,
        rng_seed,
        time_limit
    )
