import java.util.*;
import java.io.*;
import com.hexaly.optimizer.*;

public class SurgeriesScheduling {
    // Number of surgeries
    private int numSurgeries;
    // Number of rooms
    private int numRooms;
    // Number of nurses
    private int numNurses;
    // Minimum start of each surgery
    private int[] minStart;
    // Maximum end of each surgery
    private int[] maxEnd;
    // Duration of each surgery
    private int[] duration;
    // Number of nurses needed for each surgery
    private int [] neededNurses;
    // Earliest starting shift for each nurse
    private int [] shiftEarliestStart;
    // Latest ending shift for each nurse
    private int [] shiftLatestEnd;
    // Maximum duration of each nurse's shift
    private int maxShiftDuration;
    // Incompatible rooms for each surgery
    private int[][] incompatibleRooms;

    // Hexaly Optimizer
    final HexalyOptimizer optimizer;
    // Decision variables: time range of each surgery
    private HxExpression[] surgeries;
    // Decision variables: sequence of surgery on each room
    private HxExpression[] surgeryOrder;
    // For each surgery, the selected room
    // This variable is only used to export the solution
    private HxExpression[] selectedRoom;
    // Decision variables: the surgeries order of each nurse
    private HxExpression[] nurseOrder;
    // Objective = minimize the makespan: end of the last surgery
    private HxExpression makespan;

    public SurgeriesScheduling(HexalyOptimizer optimizer) throws IOException {
        this.optimizer = optimizer;
    }

    public void readInstance(String fileName) throws IOException {
        try (Scanner input = new Scanner(new File(fileName))) {
            input.useLocale(Locale.ROOT);
            numRooms = input.nextInt();
            numNurses = input.nextInt();
            numSurgeries = input.nextInt();

            minStart = new int[numSurgeries];
            for (int s = 0; s < numSurgeries; ++s) {
                minStart[s] = input.nextInt() * 60;
            }

            maxEnd = new int[numSurgeries];
            for (int s = 0; s < numSurgeries; ++s) {
                maxEnd[s] = input.nextInt() * 60;
            }

            duration = new int[numSurgeries];
            for (int s = 0; s < numSurgeries; ++s) {
                duration[s] = input.nextInt();
            }

            neededNurses = new int[numSurgeries];
            for (int s = 0; s < numSurgeries; ++s) {
                neededNurses[s] = input.nextInt();
            }

            shiftEarliestStart = new int[numNurses];
            for (int n = 0; n < numNurses; ++n) {
                shiftEarliestStart[n] = input.nextInt() * 60;
            }

            shiftLatestEnd = new int[numNurses];
            for (int n = 0; n < numNurses; ++n) {
                shiftLatestEnd[n] = input.nextInt() * 60;
            }

            maxShiftDuration = input.nextInt() * 60;
            
            incompatibleRooms = new int[numSurgeries][];
            for (int s = 0; s < numSurgeries; ++s) {
                incompatibleRooms[s] = new int[numRooms];
                for (int r = 0; r < numRooms; ++r) {
                    incompatibleRooms[s][r] = input.nextInt();
                }
            } 
        }
    }

    public void solve(int timeLimit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        // Sequence of surgery for each room
        surgeryOrder = new HxExpression[numRooms];
        HxExpression room = model.array();
        for (int r = 0; r < numRooms; ++r) {
            surgeryOrder[r] = model.listVar(numSurgeries);
            room.addOperand(surgeryOrder[r]);
        }

        // Each surgery is scheduled in a room
        model.constraint(model.partition(room));

        // Only compatible rooms can be selected for a surgery
        for (int s = 0; s < numSurgeries; ++s) {
            for (int r : incompatibleRooms[s]) {
                model.constraint(model.eq(model.contains(surgeryOrder[r], s), 0));
            }
        }

        // For each surgery, the selected room
        selectedRoom = new HxExpression[numSurgeries];
        for (int s = 0; s < numSurgeries; ++s) {
            selectedRoom[s] = model.find(room, s);
        }

        // Interval decisions: time range of each surgery
        surgeries = new HxExpression[numSurgeries];
        for (int s = 0; s < numSurgeries; ++s) {
            // Each surgery cannot start before and end after a certain time
            surgeries[s] = model.intervalVar(minStart[s], maxEnd[s]);

            // Each surgery has a specific duration
            model.constraint(model.eq(model.length(surgeries[s]), duration[s]));
        }

        HxExpression surgeryArray = model.array(surgeries);

        // A room can only have one surgery at a time
        for (int r = 0; r < numRooms; ++r) {
            HxExpression sequence = surgeryOrder[r];
            HxExpression sequenceLambda = model.lambdaFunction(s -> model
                        .lt(model.at(surgeryArray, model.at(sequence, s)),
                                model.at(surgeryArray, model.at(sequence, model.sum(s, 1)))));
            model.constraint(model.and(
                model.range(0, model.sub(model.count(sequence), 1)), 
                sequenceLambda));
        }
        // Sequence of surgery for each nurse
        nurseOrder = new HxExpression[numNurses];

        for (int n = 0; n < numNurses; ++n) {
            nurseOrder[n] = model.listVar(numSurgeries);
            HxExpression sequence = nurseOrder[n];
            HxExpression firstSurgeryStart = model.iif(
                model.gt(model.count(sequence),0), 
                model.start(model.at(surgeryArray,model.at(sequence,0))),
                shiftEarliestStart[n]
            );
            HxExpression lastSurgeryEnd = model.iif(
                model.gt(model.count(sequence),0), 
                model.end(model.at(surgeryArray, model.at(sequence,model.sub(model.count(sequence),1)))),
                shiftEarliestStart[n]
            );
            // Each nurse has an earliest starting shift and latest ending shift to be respected
            model.constraint(model.geq(firstSurgeryStart, shiftEarliestStart[n]));
            model.constraint(model.leq(lastSurgeryEnd, shiftLatestEnd[n]));
            // Each nurse cannot work more than a certain amount of hours in a row
            model.constraint(model.leq(
                model.sub(lastSurgeryEnd, firstSurgeryStart), 
                maxShiftDuration));
            // Each nurse can only be at one operation at a time and stays all along the surgery
            HxExpression sequenceLambda = model.lambdaFunction(s -> model
                        .lt(model.at(surgeryArray, model.at(sequence, s)),
                            model.at(surgeryArray, model.at(sequence, model.sum(s, 1)))));
            model.constraint(model.and(
                model.range(0, model.sub(model.count(sequence), 1)), sequenceLambda));

        }

        HxExpression nurseOrderArray = model.array(nurseOrder);

        // Each surgery needs a specific amount of nurse
        for (int s = 0; s < numSurgeries; ++s) {
            int surgery = s;
            model.constraint(model.geq(model.sum(
                model.range(0, numNurses), 
                model.lambdaFunction(
                    n -> model.contains(model.at(nurseOrderArray, n), surgery))), neededNurses[s]));
        }
        
        // Minimize the makespan: end of the last task
        makespan = model.max();
        for (int s = 0; s < numSurgeries; ++s) {
            makespan.addOperand(model.end(surgeries[s]));
        }
        model.minimize(makespan);

        model.close();

        // Parameterize the optimizer
        optimizer.getParam().setTimeLimit(timeLimit);

        optimizer.solve();
    }

    /*
     * Write the solution in a file with the following format:
     * - for each operation of each job, the selected machine, the start and end
     * dates, the nurses working on this operation
     */
    public void writeSolution(String fileName) throws IOException {
        try (PrintWriter output = new PrintWriter(fileName)) {
            System.out.println("Solution written in file " + fileName);

            List<List<Integer>> listNurses = new ArrayList<List<Integer>>();
            for (int s = 0; s < numSurgeries; ++s) {
                listNurses.add(new ArrayList<Integer>());
            }
            for (int n = 0; n < numNurses; ++n) {
                HxCollection surgNurse = nurseOrder[n].getCollectionValue();
                for (long s : surgNurse) {
                    listNurses.get((int) s).add(n);
                }
            }

            for (int s = 0; s < numSurgeries; ++s) {
                output.write(s + "\t"
                        + "\t" + selectedRoom[s].getValue()
                        + "\t" + surgeries[s].getIntervalValue().getStart()
                        + "\t" + surgeries[s].getIntervalValue().getEnd() 
                        + "\t" + listNurses.get(s) + "\n");
            }
        }
    }
    public static void main(String[] args) {
        if (args.length < 1) {
            System.out.println("Usage: java SurgeriesScheduling instanceFile [outputFile] [timeLimit]");
            System.exit(1);
        }

        String instanceFile = args[0];
        String outputFile = args.length > 1 ? args[1] : null;
        String strTimeLimit = args.length > 2 ? args[2] : "20";

        try (HexalyOptimizer optimizer = new HexalyOptimizer()) {
            SurgeriesScheduling model = new SurgeriesScheduling(optimizer);
            model.readInstance(instanceFile);
            model.solve(Integer.parseInt(strTimeLimit));
            if (outputFile != null) {
                model.writeSolution(outputFile);
            }
        } catch (Exception ex) {
            System.err.println(ex);
            ex.printStackTrace();
            System.exit(1);
        }
    }
}
