#include "optimizer/hexalyoptimizer.h"
#include <algorithm>
#include <fstream>
#include <iostream>
#include <limits>
#include <numeric>
#include <vector>

using namespace hexaly;

class SurgeriesScheduling {
private:
    // Number of surgeries 
    int numSurgeries;
    // Number of rooms
    int numRooms;
    // Number of nurses
    int numNurses;
    // Minimum start of each surgery
    std::vector<int> minStart;
    // Maximum end of each surgery
    std::vector<int> maxEnd;
    // Duration of each surgery
    std::vector<int> duration;
    // Number of nurses needed for each surgery
    std::vector<int> neededNurses;
    // Earliest starting shift for each nurse
    std::vector<int> shiftEarliestStart;
    // Latest ending shift for each nurse
    std::vector<int> shiftLatestEnd;
    // Maximum duration of each nurse's shift
    int maxShiftDuration;
    // Incompatible rooms for each surgery
    std::vector<std::vector<int>> incompatibleRooms;

    // Hexaly Optimizer
    HexalyOptimizer optimizer;
    // Decision variables: time range of each surgery
    std::vector<HxExpression> surgeries;
    // Decision variables: sequence of surgery on each room
    std::vector<HxExpression> surgeryOrder;
    // For each surgery, the selected room
    // This variable is only used to export the solution
    std::vector<HxExpression> selectedRoom;
    // Decision variables: for each nurse, their surgeries order
    std::vector<HxExpression> nurseOrder;
    // Objective = minimize the makespan: end of the last surgery
    HxExpression makespan;

public:
    SurgeriesScheduling() : optimizer() {}

    void readInstance(const std::string& fileName) {
        std::ifstream infile;
        infile.exceptions(std::ifstream::failbit | std::ifstream::badbit);
        infile.open(fileName.c_str());

        infile >> numRooms;
        infile >> numNurses;
        infile >> numSurgeries;

        minStart.resize(numSurgeries);
        for (int s = 0; s < numSurgeries; ++s) {
            int start;
            infile >> start;
            minStart[s] = start * 60;
        }
        
        maxEnd.resize(numSurgeries);
        for (int s = 0; s < numSurgeries; ++s) {
            int end;
            infile >> end;
            maxEnd[s] = end * 60;
        }
        
        duration.resize(numSurgeries);
        for (int s = 0; s < numSurgeries; ++s)
            infile >> duration[s];
        
        neededNurses.resize(numSurgeries);
        for (int s = 0; s < numSurgeries; ++s)
            infile >> neededNurses[s];

        shiftEarliestStart.resize(numNurses);
        for (int s = 0; s < numNurses; ++s) {
            int shift;
            infile >> shift;
            shiftEarliestStart[s] = shift * 60 ;
        }

        shiftLatestEnd.resize(numNurses);
        for (int s = 0; s < numNurses; ++s) {
            int shift;
            infile >> shift;
            shiftLatestEnd[s] = shift * 60;
        }
        int maxShiftDurationHeure;
        infile >> maxShiftDurationHeure;
        maxShiftDuration = maxShiftDurationHeure * 60;

        incompatibleRooms.resize(numSurgeries);
        for (int s = 0; s < numSurgeries; ++s) {
            incompatibleRooms[s].resize(numRooms);
            for (int r = 0; r < numRooms; ++r) {
                int room;
                infile >> room;
                incompatibleRooms[s][r] = room;
            }
        }
        infile.close();
    }

    void solve(int timeLimit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        // Sequence of surgery on each room
        surgeryOrder.resize(numRooms);
        HxExpression room = model.array();
        for (unsigned int r = 0; r < numRooms; ++r) {
            surgeryOrder[r] = model.listVar(numSurgeries);
            room.addOperand(surgeryOrder[r]);
        }

        // Each surgery is scheduled in a room
        model.constraint(model.partition(room));

        // Only compatible rooms can be selected for a surgery
        for (int s = 0; s < numSurgeries; ++s) {
            for (int r : incompatibleRooms[s]) {
                model.constraint(model.contains(surgeryOrder[r], s) == 0);
            }
        }

        // For each surgery, the selected room
        selectedRoom.resize(numSurgeries);
        for (int s = 0; s < numSurgeries; ++s) {
            selectedRoom[s] = model.find(room, s);
        }

        // Interval decisions: time range of each surgery
        surgeries.resize(numSurgeries);

        for (int s = 0; s < numSurgeries; ++s) {
            // Each surgery cannot start before and end after a certain time
            surgeries[s] = model.intervalVar(minStart[s], maxEnd[s]);

            // Each surgery has a specific duration
            model.constraint(model.length(surgeries[s]) == duration[s]);
        }
        HxExpression surgeryArray = model.array(surgeries.begin(), surgeries.end());

        // A room can only have one surgery at a time
        for (int r = 0; r < numRooms; ++r) {
            HxExpression sequence = surgeryOrder[r];
            HxExpression sequenceLambda = model.createLambdaFunction([&](HxExpression s) { 
                return surgeryArray[sequence[s]] < surgeryArray[sequence[s + 1]]; });
            model.constraint(model.and_(model.range(0, model.count(sequence) - 1), sequenceLambda));
        }

        // Sequence of surgery for each nurse
        nurseOrder.resize(numNurses);
        
        for (int n = 0; n < numNurses; ++n) {
            nurseOrder[n] = model.listVar(numSurgeries);
            HxExpression sequence = nurseOrder[n];
            HxExpression firstSurgeryStart = model.iif(
                model.count(sequence) > 0, 
                model.start(surgeryArray[sequence[0]]),
                shiftEarliestStart[n]);
            HxExpression lastSurgeryEnd = model.iif(
                model.count(sequence) > 0,
                model.end(surgeryArray[sequence[model.count(sequence)-1]]),
                shiftEarliestStart[n]);

            // Each nurse has an earliest starting shift and latest ending shift to be respected
            model.constraint(firstSurgeryStart >= shiftEarliestStart[n]);
            model.constraint(lastSurgeryEnd <= shiftLatestEnd[n]);

            // Each nurse cannot work more than a certain amount of hours in a row
            model.constraint(lastSurgeryEnd - firstSurgeryStart <= maxShiftDuration);

            // Each nurse can only be at one operation at a time and stays all along the surgery
            HxExpression sequenceLambda = model.createLambdaFunction([&](HxExpression s) { 
                return surgeryArray[sequence[s]] < surgeryArray[sequence[s + 1]]; });
            model.constraint(model.and_(model.range(0, model.count(sequence) - 1), sequenceLambda));
        }
        
        HxExpression nurseOrderArray = model.array(nurseOrder.begin(), nurseOrder.end());
        
        // Each surgery needs a specific amount of nurse
        for (int s = 0; s < numSurgeries; ++s) {
            model.constraint(model.sum(model.range(0, numNurses), 
                    model.createLambdaFunction([&](HxExpression n) { 
                        return model.contains(nurseOrderArray[n], s);})) 
                >= neededNurses[s]);
        } 

        // Minimize the makespan: end of the last surgery
        makespan = model.max();
        for (int s = 0; s < numSurgeries; ++s) {
            makespan.addOperand(model.end(surgeries[s]));
        }
        model.minimize(makespan);

        model.close();

        // Parameterize the optimizer
        optimizer.getParam().setTimeLimit(timeLimit);

        optimizer.solve();
    }

    /* Write the solution in a file with the following format:
     *  - for each surgery, the selected room, the start and end dates, 
          the nurses working on this operation */
    void writeSolution(const std::string& fileName) {
        std::ofstream outfile(fileName.c_str());
        if (!outfile.is_open()) {
            std::cerr << "File " << fileName << " cannot be opened." << std::endl;
            exit(1);
        }
        std::cout << "Solution written in file " << fileName << std::endl;

        std::vector<std::vector<int>> listNurses;
        listNurses.resize(numSurgeries);
        for (unsigned int n = 0; n < numNurses; ++n) {
            HxCollection surgNurse = nurseOrder[n].getCollectionValue();
            for (int s = 0; s < surgNurse.count(); ++s) {
                listNurses[surgNurse.get(s)].push_back(n);
            }
        }

        for (unsigned int s = 0; s < numSurgeries; ++s) {
            outfile << s << "\t" << selectedRoom[s].getValue() << "\t"
                    << surgeries[s].getIntervalValue().getStart() << "\t"
                    << surgeries[s].getIntervalValue().getEnd() << "\t";
            outfile << "[";
            bool first = true;
            for (int n : listNurses[s]) {
                if (!first) {
                    outfile << ", ";
                }
                outfile << n;
                first = false;
            }
            outfile << "]" << std::endl;
        }
        outfile.close();
    }
};

int main(int argc, char** argv) {
    if (argc < 2) {
        std::cout << "Usage: surgeries_scheduling instanceFile [outputFile] [timeLimit]" << std::endl;
        exit(1);
    }

    const char* instanceFile = argv[1];
    const char* outputFile = argc > 2 ? argv[2] : NULL;
    const char* strTimeLimit = argc > 3 ? argv[3] : "20";

    SurgeriesScheduling model;
    try {
        model.readInstance(instanceFile);
        const int timeLimit = atoi(strTimeLimit);
        model.solve(timeLimit);
        if (outputFile != NULL)
            model.writeSolution(outputFile);
        return 0;
    } catch (const std::exception& e) {
        std::cerr << "An error occurred: " << e.what() << std::endl;
        return 1;
    }
}