import hexaly.optimizer
import sys

def read_instance(filename):
    with open(filename) as f:
        lines = f.readlines()

    first_line = lines[0].split()
    # Number of rooms
    num_rooms = int(first_line[0])
    # Number of nurses
    num_nurses = int(first_line[1])
    # Number of surgeries
    num_surgeries = int(first_line[2])

    # Minimum start of each surgery
    line_min_start = lines[1].split()
    min_start = [int(line_min_start[s]) * 60 for s in range(num_surgeries)]

    # Maximum end of each surgery
    line_max_end = lines[2].split()
    max_end = [int(line_max_end[s]) * 60 for s in range(num_surgeries)]

    # Duration of each surgery
    line_duration = lines[3].split()
    duration = [int(line_duration[s]) for s in range(num_surgeries)]

    # Number of nurses needed for each surgery
    line_nurse_needed = lines[4].split()
    needed_nurses = [int(line_nurse_needed[s]) for s in range(num_surgeries)]

    # Earliest starting shift for each nurse
    line_earliest_shift = lines[5].split()
    shift_earliest_start = [int(line_earliest_shift[s]) * 60 for s in range(num_nurses)]

    # Latest ending shift for each nurse
    line_latest_shift = lines[6].split()
    shift_latest_end = [int(line_latest_shift[s]) * 60 for s in range(num_nurses)]

    # Maximum duration of each nurse's shift
    max_shift_duration = int(lines[7].split()[0]) * 60
    
    #Incompatible rooms for each surgery
    incompatible_rooms = [[0 for r in range(num_rooms)] for s in range(num_surgeries)]
    for s in range(num_surgeries):
        line = lines[8+s].split()
        for r in range(num_rooms):
            incompatible_rooms[s][r] = int(line[r])

    return (num_rooms, num_nurses, num_surgeries, min_start, max_end, 
            needed_nurses, shift_earliest_start, shift_latest_end, 
            max_shift_duration, incompatible_rooms, duration)

def main(instance_file, output_file, time_limit):
    num_rooms, num_nurses, num_surgeries, min_start, max_end, needed_nurses, \
    shift_earliest_start, shift_latest_end, max_shift_duration, \
    incompatible_rooms, duration = read_instance(instance_file)
    
    with hexaly.optimizer.HexalyOptimizer() as optimizer:
        #
        # Declare the optimization model
        #
        model = optimizer.model

        # Sequence of surgery for each room
        surgery_order = [model.list(num_surgeries) for _ in range(num_rooms)]
        rooms = model.array(surgery_order)

        # Each surgery is scheduled in a room
        model.constraint(model.partition(rooms))

        # Only compatible rooms can be selected for a surgery
        for s in range(num_surgeries):
            for r in incompatible_rooms[s]:
                model.constraint(model.contains(surgery_order[r], s) == 0)

        # For each surgery, the selected room
        # This variable is only used to export the solution
        selected_room = [model.find(rooms, s) for s in range(num_surgeries)]

        # Interval decisions: time range of each surgery
        # Each surgery cannot start before and end after a certain time
        surgeries = [model.interval(min_start[s], max_end[s]) for s in range(num_surgeries)]

        for s in range(num_surgeries):
            # Each surgery has a specific duration
            model.constraint(model.length(surgeries[s]) == duration[s])

        surgery_array = model.array(surgeries)

        # A room can only have one surgery at a time
        for r in range(num_rooms):
            sequence = surgery_order[r]
            sequence_lambda = model.lambda_function(
                lambda s: surgery_array[sequence[s]] < surgery_array[sequence[s + 1]])
            model.constraint(
                model.and_(model.range(0, model.count(sequence) - 1), sequence_lambda)
            )

        # Each surgery needs a specific amount of nurse
        nurse_order = [model.list(num_surgeries) for _ in range(num_nurses)]

        for n in range(num_nurses):
            # Each nurse has an earliest starting shift and latest ending shift to be respected
            sequence = nurse_order[n]
            first_surgery_start = model.iif( 
                model.count(sequence) > 0, 
                model.start(surgery_array[sequence[0]]),
                shift_earliest_start[n]
            )
            last_surgery_end = model.iif(
                model.count(sequence) > 0, 
                model.end(surgery_array[sequence[model.count(sequence)-1]]),
                shift_earliest_start[n]
            )
            
            model.constraint(first_surgery_start >= shift_earliest_start[n])
            model.constraint(last_surgery_end <= shift_latest_end[n])

            # Each nurse cannot work more than a certain amount of hours
            model.constraint(last_surgery_end - first_surgery_start <= max_shift_duration)
            
            # Each nurse can only be at one operation at a time and stays all along the surgery
            sequence_lambda = model.lambda_function(
                lambda s: surgery_array[sequence[s]] < surgery_array[sequence[s + 1]])
            model.constraint(model.and_(
                model.range(0, model.count(sequence) - 1), sequence_lambda))
        
        # Each surgery needs a certain amount of nurses 
        nurse_order_array = model.array(nurse_order)
        for s in range(num_surgeries):
            model.constraint(
                model.sum(
                    model.range(0, num_nurses), 
                    model.lambda_function(lambda n : model.contains(nurse_order_array[n], s))
                )
                >= needed_nurses[s]
            )

        # Minimize the makespan: end of the last task
        makespan = model.max([model.end(surgeries[s]) for s in range(num_surgeries)])
        model.minimize(makespan)

        model.close()

        # Parameterize the optimizer
        optimizer.param.time_limit = time_limit

        optimizer.solve()

        # Write the solution in a file with the following format:
        # - for each surgery, the selected room, the start and end dates, 
        # the nurses working on this operation
        if output_file != None:
            with open(output_file, "w") as f:
                print("Solution written in file", output_file)
                list_nurses = {}
                for n in range(num_nurses):
                    surg_nurse = nurse_order[n].value
                    for s in surg_nurse:
                        if s not in list_nurses:
                            list_nurses[s] = [n]
                        else:
                            list_nurses[s].append(n)
                for s in range(num_surgeries):
                    f.write(str(s) + "\t"
                        + "\t" + str(selected_room[s].value)
                        + "\t" + str(surgeries[s].value.start())
                        + "\t" + str(surgeries[s].value.end()) 
                        + "\t" + str(list_nurses[s]) + "\n")



if __name__ == '__main__':
    if len(sys.argv) < 2:
        print("Usage: python surgeries_scheduling.py instance_file [output_file] [time_limit]")
        sys.exit(1)

    instance_file = sys.argv[1]
    output_file = sys.argv[2] if len(sys.argv) >= 3 else None
    time_limit = int(sys.argv[3]) if len(sys.argv) >= 4 else 20
    main(instance_file, output_file, time_limit)


