import java.util.*;

import org.w3c.dom.ls.LSException;

import java.io.*;
import com.hexaly.optimizer.*;

public class VehicleRoutingWithBackhauls {
    // Hexaly Optimizer
    private final HexalyOptimizer optimizer;

    // Number of customers
    int nbCustomers;

    // Capacity of the trucks
    private int truckCapacity;

    // Demand on each customer
    private long[] deliveryDemandsData;
    private long[] pickupDemandsData;

    // Type of each customer
    private int[] isBackhaulData;

    // Distance matrix between customers
    private long[][] distMatrixData;

    // Distances between customers and depot
    private long[] distDepotData;

    // Number of trucks
    private int nbTrucks;

    // Decision variables
    private HxExpression[] customersSequences;

    // Are the trucks actually used
    private HxExpression[] trucksUsed;

    // Number of trucks used in the solution
    private HxExpression nbTrucksUsed;

    // Distance traveled by all the trucks
    private HxExpression totalDistance;

    private VehicleRoutingWithBackhauls(HexalyOptimizer optimizer) {
        this.optimizer = optimizer;
    }

    private void solve(int limit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        trucksUsed = new HxExpression[nbTrucks];
        customersSequences = new HxExpression[nbTrucks];
        HxExpression[] distRoutes = new HxExpression[nbTrucks];

        // Sequence of customers visited by each truck
        for (int k = 0; k < nbTrucks; ++k)
            customersSequences[k] = model.listVar(nbCustomers);

        // All customers must be visited by exactly one truck
        model.constraint(model.partition(customersSequences));

        // Create HexalyOptimizer arrays to be able to access them with an "at" operator
        HxExpression deliveryDemands = model.array(deliveryDemandsData);
        HxExpression pickupDemands = model.array(pickupDemandsData);
        HxExpression isBackhaul = model.array(isBackhaulData);
        HxExpression distDepot = model.array(distDepotData);
        HxExpression distMatrix = model.array(distMatrixData);

        for (int k = 0; k < nbTrucks; ++k) {
            HxExpression sequence = customersSequences[k];
            HxExpression c = model.count(sequence);

            // A truck is used if it visits at least one customer
            trucksUsed[k] = model.gt(c, 0);

            // A pickup cannot be followed by a delivery
            HxExpression precedencyLambda = model.lambdaFunction(
                    i -> model.leq(
                            model.at(isBackhaul, model.at(sequence, model.sub(i, 1))),
                            model.at(isBackhaul, model.at(sequence, i))));
            model.constraint(model.and(model.range(1, c), precedencyLambda));

            // The quantity needed in each route must not exceed the truck capacity
            HxExpression deliveryDemandLambda = model.lambdaFunction(j -> model.at(deliveryDemands, j));
            HxExpression routeDeliveryQuantity = model.sum(sequence, deliveryDemandLambda);
            model.constraint(model.leq(routeDeliveryQuantity, truckCapacity));
            HxExpression pickupDemandLambda = model.lambdaFunction(j -> model.at(pickupDemands, j));
            HxExpression routePickupQuantity = model.sum(sequence, pickupDemandLambda);
            model.constraint(model.leq(routePickupQuantity, truckCapacity));

            // Distance traveled by truck k
            HxExpression distLambda = model
                    .lambdaFunction(
                            i -> model.at(distMatrix, model.at(sequence, model.sub(i, 1)), model.at(sequence, i)));
            distRoutes[k] = model.sum(model.sum(model.range(1, c), distLambda),
                    model.iif(model.gt(c, 0), model.sum(model.at(distDepot, model.at(sequence, 0)),
                            model.at(distDepot, model.at(sequence, model.sub(c, 1)))), 0));
        }

        nbTrucksUsed = model.sum(trucksUsed);
        totalDistance = model.sum(distRoutes);

        // Objective: minimize the number of trucks used, then minimize the distance
        // traveled
        model.minimize(nbTrucksUsed);
        model.minimize(totalDistance);

        model.close();

        // Parametrize the optimizer
        optimizer.getParam().setTimeLimit(limit);

        optimizer.solve();
    }

    /*
     * Write the solution in a file with the following format:
     * - number of trucks used and total distance
     * - for each truck the customers visited (omitting the start/end at the depot)
     */
    private void writeSolution(String fileName) throws IOException {
        try (PrintWriter output = new PrintWriter(fileName)) {
            output.println(nbTrucksUsed.getValue() + " " + totalDistance.getValue());
            for (int k = 0; k < nbTrucks; ++k) {
                if (trucksUsed[k].getValue() != 1)
                    continue;
                // Values in sequence are in 0...nbCustomers. +2 is to put it back in
                // 2...nbCustomers as in the data files (1 being the depot)
                HxCollection customersCollection = customersSequences[k].getCollectionValue();
                for (int i = 0; i < customersCollection.count(); ++i) {
                    output.print((customersCollection.get(i) + 2) + " ");
                }
                output.println();
            }
        }
    }

    // The input files follow the "CVRPLib" format
    private void readInstance(String fileName) throws IOException {
        try (Scanner input = new Scanner(new File(fileName))) {
            input.useLocale(Locale.ROOT);
            int nbNodes = 0;
            String[] splitted;
            while (true) {
                splitted = input.nextLine().split(":");
                if (splitted[0].contains("DIMENSION")) {
                    nbNodes = Integer.parseInt(splitted[1].trim());
                    nbCustomers = nbNodes - 1;
                } else if (splitted[0].contains("VEHICLES")) {
                    nbTrucks = Integer.parseInt(splitted[1].trim());
                } else if (splitted[0].contains("CAPACITY")) {
                    truckCapacity = Integer.parseInt(splitted[1].trim());
                } else if (splitted[0].contains("EDGE_WEIGHT_TYPE")) {
                    if (splitted[1].trim().compareTo("EXACT_2D") != 0) {
                        throw new RuntimeException(
                                "Edge Weight Type " + splitted[1] + " is not supported (only EXACT_2D)");
                    }
                } else if (splitted[0].contains("NODE_COORD_SECTION")) {
                    break;
                }
            }

            int[] customersX = new int[nbCustomers];
            int[] customersY = new int[nbCustomers];
            int depotX = 0, depotY = 0;
            for (int n = 1; n <= nbNodes; ++n) {
                int id = input.nextInt();
                if (id != n)
                    throw new IOException("Unexpected index");
                if (n == 1) {
                    depotX = input.nextInt();
                    depotY = input.nextInt();
                } else {
                    // -2 because original customer indices are in 2..nbNodes
                    customersX[n - 2] = input.nextInt();
                    customersY[n - 2] = input.nextInt();
                }
            }

            computeDistanceMatrix(depotX, depotY, customersX, customersY);

            input.nextLine().split(":"); // End the last line
            splitted = input.nextLine().split(":");
            if (!splitted[0].contains("DEMAND_SECTION")) {
                throw new RuntimeException("Expected keyword DEMAND_SECTION");
            }

            long[] demandsData = new long[nbCustomers];
            for (int n = 1; n <= nbNodes; ++n) {
                int id = input.nextInt();
                if (id != n)
                    throw new IOException("Unexpected index");
                int demand = input.nextInt();
                if (n == 1) {
                    if (demand != 0)
                        throw new IOException("Depot demand is supposed to be 0");
                } else {
                    // -2 because original customer indices are in 2..nbNodes
                    demandsData[n - 2] = demand;
                }
            }

            input.nextLine().split(":"); // End the last line
            splitted = input.nextLine().split(":");
            if (!splitted[0].contains("BACKHAUL_SECTION")) {
                throw new RuntimeException("Expected keyword DEPOT_SECTION");
            }

            isBackhaulData = new int[nbCustomers];
            while (true) {
                int id = input.nextInt();
                if (id == -1)
                    break;
                // -2 because original customer indices are in 2..nbNodes
                isBackhaulData[id - 2] = 1;
            }

            deliveryDemandsData = new long[nbCustomers];
            pickupDemandsData = new long[nbCustomers];
            for (int i = 0; i < nbCustomers; ++i) {
                if (isBackhaulData[i] == 1) {
                    pickupDemandsData[i] = demandsData[i];
                } else {
                    deliveryDemandsData[i] = demandsData[i];
                }
            }

            input.nextLine().split(":"); // End the last line
            splitted = input.nextLine().split(":");
            if (!splitted[0].contains("DEPOT_SECTION")) {
                throw new RuntimeException("Expected keyword DEPOT_SECTION");
            }

            int depotId = input.nextInt();
            if (depotId != 1)
                throw new IOException("Depot id is supposed to be 1");

            int endOfDepotSection = input.nextInt();
            if (endOfDepotSection != -1) {
                throw new RuntimeException("Expecting only one depot, more than one found");
            }
        }
    }

    // Compute the distance matrix
    private void computeDistanceMatrix(int depotX, int depotY, int[] customersX, int[] customersY) {
        distMatrixData = new long[nbCustomers][nbCustomers];
        for (int i = 0; i < nbCustomers; ++i) {
            distMatrixData[i][i] = 0;
            for (int j = i + 1; j < nbCustomers; ++j) {
                long dist = computeDist(customersX[i], customersX[j], customersY[i], customersY[j]);
                distMatrixData[i][j] = dist;
                distMatrixData[j][i] = dist;
            }
        }

        distDepotData = new long[nbCustomers];
        for (int i = 0; i < nbCustomers; ++i) {
            distDepotData[i] = computeDist(depotX, customersX[i], depotY, customersY[i]);
        }
    }

    private long computeDist(int xi, int xj, int yi, int yj) {
        double exactDist = Math.sqrt(Math.pow(xi - xj, 2) + Math.pow(yi - yj, 2));
        return Math.round(exactDist);
    }

    public static void main(String[] args) {
        if (args.length < 1) {
            System.err.println("Usage: java VehicleRoutingWithBackhauls inputFile [outputFile] [timeLimit]");
            System.exit(1);
        }

        try (HexalyOptimizer optimizer = new HexalyOptimizer()) {
            String instanceFile = args[0];
            String outputFile = args.length > 1 ? args[1] : null;
            String strTimeLimit = args.length > 2 ? args[2] : "20";

            VehicleRoutingWithBackhauls model = new VehicleRoutingWithBackhauls(optimizer);
            model.readInstance(instanceFile);
            model.solve(Integer.parseInt(strTimeLimit));
            if (outputFile != null) {
                model.writeSolution(outputFile);
            }
        } catch (Exception ex) {
            System.err.println(ex);
            ex.printStackTrace();
            System.exit(1);
        }
    }
}
