import com.hexaly.optimizer.*;
import java.io.*;
import java.util.*;

public class Vrptf {
    // Hexaly Optimizer
    private final HexalyOptimizer optimizer;

    // Number of customers
    private int nbCustomers;

    // Customers coordinates
    private int[] xCustomers;
    private int[] yCustomers;

    // Customers demands
    private int[] demandsData;

    // Number of facilities
    private int nbFacilities;

    // Facilities coordinates
    private int[] xFacilities;
    private int[] yFacilities;

    // Number of points
    private int nbPoints;

    // Depot coordinates
    private int xDepot;
    private int yDepot;

    // Number of trucks
    private int nbTrucks;

    // Capacity of trucks
    private int truckCapacity;

    // Distance matrix
    private long[][] distMatrixData;

    // Distance to depot array
    private long[] distDepotsData;

    // Assignement costs
    private long[] assignmentCostsData;

    // Decision variables
    private HxExpression[] routesSequences;

    // Objective value
    private HxExpression totalDistanceCost;
    private HxExpression totalAssignementCost;
    private HxExpression totalCost;

    private Vrptf(HexalyOptimizer optimizer) {
        this.optimizer = optimizer;
    }

    private void solve(int limit) {
        // Declare the optimization model
        HxModel m = optimizer.getModel();

        double totalDemand = 0;
        for (int c = 0; c < nbCustomers; ++c) {
            totalDemand += demandsData[c];
        }
        int minNbTrucks = (int) Math.ceil(totalDemand / truckCapacity);
        nbTrucks = (int) Math.ceil(1.5 * minNbTrucks);

        routesSequences = new HxExpression[nbTrucks];

        // A route is represented as a list containing the points in the order they are
        // visited
        for (int i = 0; i < nbTrucks; ++i) {
            routesSequences[i] = m.listVar(nbPoints);
        }

        HxExpression routes = m.array(routesSequences);

        // Each point must be visited at most once
        m.constraint(m.disjoint(routesSequences));

        HxExpression demands = m.array(demandsData);
        HxExpression distMatrix = m.array(distMatrixData);
        HxExpression distDepots = m.array(distDepotsData);
        HxExpression assignmentCosts = m.array(assignmentCostsData);

        HxExpression[] distRoutes = new HxExpression[nbTrucks];
        HxExpression[] assignmentCostRoutes = new HxExpression[nbTrucks];

        for (int c = 0; c < nbCustomers; ++c) {
            int startFacilities = nbCustomers + c * nbFacilities;
            int endFacilities = startFacilities + nbFacilities;

            // Each customer is either contained in a route or assigned to a facility
            HxExpression facilityUsedSum = m.sum();
            for (int f = startFacilities; f < endFacilities; ++f) {
                facilityUsedSum.addOperand(m.contains(routes, f));
            }
            m.constraint(m.eq(m.sum(m.contains(routes, c), facilityUsedSum), 1));
        }

        for (int r = 0; r < nbTrucks; ++r) {
            HxExpression route = routesSequences[r];
            HxExpression c = m.count(route);

            // Each truck cannot carry more than its capacity
            HxExpression demandLambda = m.lambdaFunction(j -> m.at(demands, j));
            HxExpression quantityServed = m.sum(route, demandLambda);
            m.constraint(m.leq(quantityServed, truckCapacity));

            HxExpression distLambda = m
                    .lambdaFunction(i -> m.at(distMatrix, m.at(route, i), m.at(route, m.sum(i, 1))));

            // Truck is used if it visits at least one point
            HxExpression truckUsed = m.gt(c, 0);

            // Distance traveled by each truck
            distRoutes[r] = m.sum(
                    m.sum(m.range(0, m.sub(c, 1)), distLambda),
                    m.iif(truckUsed,
                            m.sum(m.at(distDepots, m.at(route, 0)),
                                    m.at(distDepots, m.at(route, m.sub(c, 1)))),
                            0));

            // The cost to assign customers to their facility
            HxExpression assignmentCostLambda = m.lambdaFunction(i -> m.at(assignmentCosts, i));
            assignmentCostRoutes[r] = m.sum(route, assignmentCostLambda);
        }

        // The total distance travelled
        totalDistanceCost = m.sum(distRoutes);
        // The total assignement cost
        totalAssignementCost = m.sum(assignmentCostRoutes);

        // Objective: minimize the sum of the total distance travelled and the total
        // assignement cost
        totalCost = m.sum(totalDistanceCost, totalAssignementCost);
        m.minimize(totalCost);
        m.close();

        optimizer.getParam().setTimeLimit(limit);
        optimizer.solve();
    }

    /* Write the solution to a file */
    private void writeSolution(String infile, String outfile) throws IOException {
        try (PrintWriter output = new PrintWriter(outfile)) {
            output.println("File name: " + infile + "; totalCost = " + totalCost.getIntValue() + "; totalDistance = "
                    + totalDistanceCost.getIntValue() + "; totalAssignementCost = "
                    + totalAssignementCost.getIntValue());
            for (int r = 0; r < nbTrucks; ++r) {
                HxCollection route = routesSequences[r].getCollectionValue();
                if (route.count() == 0)
                    continue;
                output.print("Route " + r + " [");
                for (int i = 0; i < route.count(); i++) {
                    long point = route.get(i);
                    if (point < nbCustomers) {
                        output.print("Customer " + point);
                    } else {
                        output.print("Facility " + point % nbCustomers + " assigned to Customer "
                                + (point - nbCustomers) / nbFacilities);
                    }
                    if (i < route.count() - 1) {
                        output.print(", ");
                    }
                }
                output.println("]");
            }
        } catch (FileNotFoundException e) {
            System.out.println("An error occurred.");
            e.printStackTrace();
        }
    }

    private void readInstance(String fileName) throws IOException {
        try (Scanner input = new Scanner(new File(fileName))) {
            input.useLocale(Locale.ROOT);
            nbCustomers = input.nextInt();
            nbFacilities = input.nextInt();
            // A point is either a customer or a facility
            // Facilities are duplicated for each customer
            nbPoints = nbCustomers + nbCustomers * nbFacilities;

            xFacilities = new int[nbFacilities];
            yFacilities = new int[nbFacilities];
            for (int f = 0; f < nbFacilities; ++f) {
                xFacilities[f] = input.nextInt();
                yFacilities[f] = input.nextInt();
            }

            xCustomers = new int[nbCustomers];
            yCustomers = new int[nbCustomers];
            for (int c = 0; c < nbCustomers; ++c) {
                xCustomers[c] = input.nextInt();
                yCustomers[c] = input.nextInt();
            }

            truckCapacity = input.nextInt();

            // Facility capacities : skip
            for (int f = 0; f < nbFacilities; ++f) {
                input.next();
            }

            demandsData = new int[nbPoints];
            for (int c = 0; c < nbCustomers; ++c) {
                demandsData[c] = input.nextInt();
                for (int f = 0; f < nbFacilities; ++f) {
                    demandsData[nbCustomers + c * nbFacilities + f] = demandsData[c];
                }
            }

            computeDepotCoordinates();
            computeDistances();
            computeAssignmentCosts();
        } catch (FileNotFoundException e) {
            System.out.println("An error occurred.");
            e.printStackTrace();
        }
    }

    void computeDepotCoordinates() {
        // Compute the coordinates of the bounding box containing all of the points
        int xMin = Integer.MAX_VALUE;
        int xMax = Integer.MIN_VALUE;
        int yMin = Integer.MAX_VALUE;
        int yMax = Integer.MIN_VALUE;

        for (int c = 0; c < nbCustomers; ++c) {
            xMin = Math.min(xMin, xCustomers[c]);
        }
        for (int f = 0; f < nbFacilities; ++f) {
            xMin = Math.min(xMin, xFacilities[f]);
        }
        for (int c = 0; c < nbCustomers; ++c) {
            xMax = Math.max(xMax, xCustomers[c]);
        }
        for (int f = 0; f < nbFacilities; ++f) {
            xMax = Math.max(xMax, xFacilities[f]);
        }
        for (int c = 0; c < nbCustomers; ++c) {
            yMin = Math.min(yMin, yCustomers[c]);
        }
        for (int f = 0; f < nbFacilities; ++f) {
            yMin = Math.min(yMin, yFacilities[f]);
        }
        for (int c = 0; c < nbCustomers; ++c) {
            yMax = Math.max(yMax, yCustomers[c]);
        }
        for (int f = 0; f < nbFacilities; ++f) {
            yMax = Math.max(yMax, yFacilities[f]);
        }

        // We assume that the depot is at the center of the bounding box
        xDepot = xMin + (xMax - xMin) / 2;
        yDepot = yMin + (yMax - yMin) / 2;
    }

    void computeDistances() {
        distDepotsData = new long[nbPoints];

        // Customer to depot
        for (int c = 0; c < nbCustomers; ++c) {
            distDepotsData[c] = computeDist(xCustomers[c], xDepot, yCustomers[c], yDepot);
        }

        // Facility to depot
        for (int c = 0; c < nbCustomers; ++c) {
            for (int f = 0; f < nbFacilities; ++f) {
                distDepotsData[nbCustomers + c * nbFacilities + f] = computeDist(xFacilities[f], xDepot, yFacilities[f],
                        yDepot);
            }
        }

        distMatrixData = new long[nbPoints][];
        for (int i = 0; i < nbPoints; i++) {
            distMatrixData[i] = new long[nbPoints];
        }

        // Distances between customers
        for (int c1 = 0; c1 < nbCustomers; ++c1) {
            for (int c2 = 0; c2 < nbCustomers; ++c2) {
                long dist = computeDist(xCustomers[c1], xCustomers[c2], yCustomers[c1], yCustomers[c2]);
                distMatrixData[c1][c2] = dist;
                distMatrixData[c2][c1] = dist;
            }
        }

        // Distances between customers and facilities
        for (int c1 = 0; c1 < nbCustomers; ++c1) {
            for (int f = 0; f < nbFacilities; ++f) {
                long dist = computeDist(xFacilities[f], xCustomers[c1], yFacilities[f], yCustomers[c1]);
                for (int c2 = 0; c2 < nbCustomers; ++c2) {
                     // Index representing serving c2 through facility f
                    int facilityIndex = nbCustomers + c2 * nbFacilities + f;
                    distMatrixData[facilityIndex][c1] = dist;
                    distMatrixData[c1][facilityIndex] = dist;
                }
            }
        }

        // Distances between facilities
        for (int f1 = 0; f1 < nbFacilities; ++f1) {
            for (int f2 = 0; f2 < nbFacilities; ++f2) {
                long dist = computeDist(xFacilities[f1], xFacilities[f2], yFacilities[f1], yFacilities[f2]);
                for (int c1 = 0; c1 < nbCustomers; ++c1) {
                    // Index representing serving c1 through facility f1
                    int index1 = nbCustomers + c1 * nbFacilities + f1;
                    for (int c2 = 0; c2 < nbCustomers; ++c2) {
                        // Index representing serving c2 through facility f2
                        int index2 = nbCustomers + c2 * nbFacilities + f2;
                        distMatrixData[index1][index2] = dist;
                    }
                }
            }
        }
    }

    void computeAssignmentCosts() {
        // Compute assignment cost for each point
        assignmentCostsData = new long[nbPoints];
        for (int c = 0; c < nbCustomers; ++c) {
            assignmentCostsData[c] = 0;
            for (int f = 0; f < nbFacilities; ++f) {
                // Cost of serving customer c through facility f
                assignmentCostsData[nbCustomers + c * nbFacilities + f] =
                    distMatrixData[c][nbCustomers + c * nbFacilities + f];
            }
        }
    }

    private long computeDist(int xi, int xj, int yi, int yj) {
        double exactDist = Math.sqrt(Math.pow(xi - xj, 2) + Math.pow(yi - yj, 2));
        return Math.round(exactDist);
    }

    public static void main(String[] args) {
        if (args.length < 1) {
            System.err.println("Usage: java vrptf inputFile [outputFile] [timeLimit]");
            System.exit(1);
        }

        try (HexalyOptimizer optimizer = new HexalyOptimizer()) {
            String instanceFile = args[0];
            String strOutfile = args.length > 1 ? args[1] : null;
            String strTimeLimit = args.length > 2 ? args[2] : "20";

            Vrptf model = new Vrptf(optimizer);
            model.readInstance(instanceFile);
            model.solve(Integer.parseInt(strTimeLimit));
            if (strOutfile != null)
                model.writeSolution(instanceFile, strOutfile);
        } catch (Exception ex) {
            System.err.println(ex);
            ex.printStackTrace();
            throw new RuntimeException(ex);
        }
    }
}
