using System;
using System.IO;
using System.Collections.Generic;
using Hexaly.Optimizer;

public class WorkforceScheduling : IDisposable
{
    // Number of agents
    private int nbAgents;
    // Number of tasks
    private int nbTasks;
    // Start of the day for each agent
    private int[] startDay;
    // End of the day for each agent
    private int[] endDay;
    // Time horizon : number of days 
    private int nbDays;
    // Time horizon : number of time steps
    private int horizon;
    // Maximum and minimum worked hours per day
    private int MIN_WORKING_TIME = 4;
    private int MAX_WORKING_TIME = 8;
    // Duration of each task
    private int[] taskDuration;
    // Number of agents needed for each task at each time
    private int[,] agentsNeeded;
    // Daily disponibilities of each agent
    private int[,] dayDispo;
    // Hour disponibilities of each agent on working days
    private int[,] hourDispo;
    // The agent is available at time t
    private int[,] agentDispo;
    // The agent is able to perform the task i
    private int[,] agentSkills;
    // The agent is able to perform the task i and complete it
    // before the end of the day
    private int[,,] agentCanStart;

    // Hexaly Optimizer
    private HexalyOptimizer optimizer;
    // Decision variable
    private HxExpression[,,] agentStart;
    // Agent working time per day
    private HxExpression[,] agentWorkingTime;
    // Attributed number of agents for each task
    private HxExpression[,] attributedAgents;
    // Difference between needed number of agents and attributed number
    // of agents for each task
    private HxExpression[,] agentDiff;
    // Indicators for lack and excess of agents
    private HxExpression agentLack;
    private HxExpression agentExcess;
    // Difference between first and last hour of agents each day
    private HxExpression[,] agentDayStart;
    private HxExpression[,] agentDayEnd;
    private HxExpression agentDayLength;

    public WorkforceScheduling()
    {
        optimizer = new HexalyOptimizer();
    }

    public void ReadInstance(string fileName)
    {
        using (StreamReader input = new StreamReader(fileName))
        {

            nbAgents = int.Parse(input.ReadLine());
            nbTasks = int.Parse(input.ReadLine());
            nbDays = int.Parse(input.ReadLine());
            horizon = nbDays * 24;

            // The duration of each task
            input.ReadLine();
            taskDuration = new int[nbTasks];
            for (int i = 0; i < nbTasks; ++i)
            {
                taskDuration[i] = int.Parse(input.ReadLine());
            }

            // Number of agents needed for the task i at time t
            // agentsNeeded[i][t] contains the number of agents needed
            // for the task i at time t
            input.ReadLine();
            input.ReadLine();
            agentsNeeded = new int[nbTasks, horizon];
            for (int d = 0; d < nbDays; ++d)
            {
                for (int i = 0; i < nbTasks; ++i)
                {
                    string[] splitted = input.ReadLine().Split(' ');
                    for (int h = 0; h < 24; ++h)
                    {
                        int t = d * 24 + h;
                        agentsNeeded[i, t] = int.Parse(splitted[h]);
                    }
                }
                input.ReadLine();
            }

            // Agent disponibility
            // dayDispo[a][d] = 1 if agent a is available on day d
            dayDispo = new int[nbAgents, nbDays];
            for (int a = 0; a < nbAgents; ++a)
            {
                string[] splitted = input.ReadLine().Split(' ');
                for (int d = 0; d < nbDays; ++d)
                {
                    dayDispo[a, d] = int.Parse(splitted[d]);
                }
            }

            // Hour disponibility
            input.ReadLine();
            hourDispo = new int[nbAgents, 24];
            startDay = new int[nbAgents];
            endDay = new int[nbAgents];
            for (int a = 0; a < nbAgents; ++a)
            {
                string[] splitted = input.ReadLine().Split(' ');
                startDay[a] = int.Parse(splitted[0]);
                endDay[a] = int.Parse(splitted[1]);
                for (int h = 0; h < 24; ++h)
                {
                    if ((h >= startDay[a]) && (h < endDay[a]))
                    {
                        hourDispo[a, h] = 1;
                    }
                    else
                    {
                        hourDispo[a, h] = 0;
                    }
                }
            }

            // We can concatenate these two informations
            // into a global indicator of agent disponibility
            // agentDispo[a][t] = 1 if agent a is available
            // on time t
            agentDispo = new int[nbAgents, horizon];
            for (int a = 0; a < nbAgents; ++a)
            {
                for (int d = 0; d < nbDays; ++d)
                {
                    for (int h = 0; h < 24; ++h)
                    {
                        int t = 24 * d + h;
                        agentDispo[a, t] = dayDispo[a, d] * hourDispo[a, h];
                    }
                }
            }

            // Agent skills
            // agentSkills[a][i] = 1 if agent a is able to perform the task i
            input.ReadLine();
            agentSkills = new int[nbAgents, nbTasks];
            for (int a = 0; a < nbAgents; ++a)
            {
                string[] splitted = input.ReadLine().Split(' ');
                for (int i = 0; i < nbTasks; ++i)
                {
                    agentSkills[a, i] = int.Parse(splitted[i]);
                }
            }

            // We can use all these informations and task length to get an
            // indicator telling us if the agent a can begin the task i
            // at time t, and complete it before the end of day
            agentCanStart = new int[nbAgents, nbTasks, horizon];
            for (int a = 0; a < nbAgents; ++a)
            {
                for (int i = 0; i < nbTasks; ++i)
                {
                    for (int d = 0; d < nbDays; ++d)
                    {
                        for (int h = 0; h < startDay[a]; ++h)
                        {
                            int t = 24 * d + h;
                            agentCanStart[a, i, t] = 0;
                        }
                        int endPossible = endDay[a] - taskDuration[i] + 1;
                        for (int h = startDay[a]; h < endPossible; ++h)
                        {
                            int t = 24 * d + h;
                            agentCanStart[a, i, t] = 
                                agentDispo[a, t] * agentSkills[a, i];
                        }
                        for (int h = endPossible; h < 24; ++h)
                        {
                            int t = 24 * d + h;
                            agentCanStart[a, i, t] = 0;
                        }
                    }
                }
            }

        }
    }

    public void Dispose()
    {
        optimizer.Dispose();
    }

    // Returns the first time step s such that
    // [s, s + duration) intersects time t
    public int intersectionStart(int t, int duration)
    {
        return Math.Max(t - duration + 1, 0);
    }

    public void Solve(int timeLimit)
    {
        // Declare the optimization model
        HxModel model = optimizer.GetModel();

        // Definition of the decision variable : 
        // agentStart[a][i][t] = 1 if agent a starts the task i at time t
        agentStart = new HxExpression[nbAgents, nbTasks, horizon];
        for (int a = 0; a < nbAgents; ++a)
        {
            for (int i = 0; i < nbTasks; ++i)
            {
                for (int t = 0; t < horizon; ++t)
                {
                    agentStart[a, i, t] = model.Bool();
                }
            }
        }

        // An agent can only work if he is able to complete his task
        // before the end of the day
        for (int a = 0; a < nbAgents; ++a)
        {
            for (int i = 0; i < nbTasks; ++i)
            {
                for (int t = 0; t < horizon; ++t)
                {
                    model.Constraint(
                        agentStart[a, i, t] <= agentCanStart[a, i, t]
                    );
                }
            }
        }

        // Each agent can only work on one task at a time
        for (int a = 0; a < nbAgents; ++a)
        {
            for (int i1 = 0; i1 < nbTasks; ++i1)
            {
                for (int i2 = 0; i2 < nbTasks; ++i2)
                {
                    for (int t1 = 0; t1 < horizon; ++t1)
                    {
                        int t2Start = intersectionStart(t1, taskDuration[i2]);
                        for (int t2 = t2Start; t2 < t1 + 1; ++t2)
                        {
                            if ((i1 != i2) || (t1 != t2))
                            {
                                model.Constraint(
                                    agentStart[a, i1, t1] +
                                    agentStart[a, i2, t2] <= 1
                                );
                            }
                        }
                    }
                }
            }
        }

        // Working time per agent
        agentWorkingTime = new HxExpression[nbAgents, nbDays];
        for (int a = 0; a < nbAgents; ++a)
        {
            for (int d = 0; d < nbDays; ++d)
            {
                agentWorkingTime[a, d] = model.Sum();
                for (int i = 0; i < nbTasks; ++i)
                {
                    for (int t = 24 * d; t < 24 * (d + 1); ++t)
                    {
                        agentWorkingTime[a, d].AddOperand(
                            agentStart[a, i, t] * taskDuration[i]
                        );
                    }
                }
            }
        }

        // Minimum and maximum amount of time worked
        for (int a = 0; a < nbAgents; ++a)
        {
            for (int d = 0; d < nbDays; ++d)
            {
                if (dayDispo[a, d] == 1)
                {
                    model.Constraint(
                        agentWorkingTime[a, d] >= MIN_WORKING_TIME
                    );
                    model.Constraint(
                        agentWorkingTime[a, d] <= MAX_WORKING_TIME
                    );
                }
            }
        }

        // Difference between needed and attributed number 
        // of agents for each task
        agentDiff = new HxExpression[nbTasks, horizon];
        attributedAgents = new HxExpression[nbTasks, horizon];
        for (int i = 0; i < nbTasks; ++i)
        {
            for (int t0 = 0; t0 < horizon; ++t0)
            {
                attributedAgents[i, t0] = model.Sum();
                for (int a = 0; a < nbAgents; ++a)
                {
                    int tStart = intersectionStart(t0, taskDuration[i]);
                    for (int t = tStart; t < t0 + 1; ++t)
                    {
                        attributedAgents[i, t0].AddOperand(agentStart[a, i, t]);
                    }
                }
                agentDiff[i, t0] = 
                    agentsNeeded[i, t0] - attributedAgents[i, t0];
            }
        }

        // Indicators for lack and excess of agents
        // Agent Lack
        agentLack = model.Sum();
        for (int i = 0; i < nbTasks; ++i)
        {
            for (int t = 0; t < horizon; ++t)
            {
                HxExpression lack = model.Max(agentDiff[i, t], 0);
                agentLack.AddOperand(model.Pow(lack, 2));
            }
        }

        // Agent Excess
        agentExcess = model.Sum();
        for (int i = 0; i < nbTasks; ++i)
        {
            for (int t = 0; t < horizon; ++t)
            {
                HxExpression excess = model.Max(-1 * agentDiff[i, t], 0);
                agentExcess.AddOperand(model.Pow(excess, 2));
            }
        }

        // Difference between first and last hour of agents each day
        agentDayLength = model.Sum();
        for (int a = 0; a < nbAgents; ++a)
        {
            for (int d = 0; d < nbDays; ++d)
            {
                HxExpression endWork = model.Max();
                HxExpression startWork = model.Min();
                for (int i = 0; i < nbTasks; ++i)
                {
                    for (int t = 24 * d; t < 24 * (d + 1); ++t)
                    {
                        endWork.AddOperand(
                            (t + taskDuration[i]) * agentStart[a, i, t]
                        );
                        startWork.AddOperand(
                            t + (1 - agentStart[a, i, t]) * horizon
                        );
                    }
                }

                HxExpression dayLength = model.Max(endWork - startWork, 0);
                agentDayLength.AddOperand(dayLength);
            }
        }

        // Minimize the agent lack, agent excess and agent day length
        model.Minimize(agentLack);
        model.Minimize(agentExcess);
        model.Minimize(agentDayLength);

        model.Close();

        // Parameterize the optimizer
        optimizer.GetParam().SetTimeLimit(timeLimit);

        optimizer.Solve();
    }

    public void WriteSolution(string fileName)
    {
        using (StreamWriter output = new StreamWriter(fileName))
        {
            Console.WriteLine("Solution written in file " + fileName);

            output.WriteLine(agentLack.GetDoubleValue());
            output.WriteLine(agentExcess.GetDoubleValue());
            output.WriteLine(agentDayLength.GetValue());
            for (int a = 0; a < nbAgents; ++a)
            {
                for (int i = 0; i < nbTasks; ++i)
                {
                    for (int t = 0; t < horizon; ++t)
                    {
                        output.Write(agentStart[a, i, t].GetValue() + " ");
                    }
                    output.WriteLine();
                }
                output.WriteLine();
            }
        }
    }

    public static void Main(string[] args)
    {
        if (args.Length < 1)
        {
            Console.WriteLine("Usage: WorkforceScheduling inputFile " +
                "[solFile] [timeLimit]");
            Environment.Exit(1);
        }
        string instanceFile = args[0];
        string outputFile = args.Length > 1 ? args[1] : null;
        string strTimeLimit = args.Length > 2 ? args[2] : "30";

        using (WorkforceScheduling model = new WorkforceScheduling())
        {
            model.ReadInstance(instanceFile);
            model.Solve(int.Parse(strTimeLimit));
            if (outputFile != null)
                model.WriteSolution(outputFile);
        }
    }
}