import java.util.*;
import java.io.*;
import com.hexaly.optimizer.*;

public class WorkforceScheduling {
    // Number of agents
    private int nbAgents;
    // Number of tasks
    private int nbTasks;
    // Start of the day for each agent
    private int[] startDay;
    // End of the day for each agent
    private int[] endDay;
    // Time horizon : number of days
    private int nbDays;
    // Time horizon : number of time steps
    private int horizon;
    // Maximum and minimum worked hours per day
    private int MIN_WORKING_TIME = 4;
    private int MAX_WORKING_TIME = 8;
    // Duration of each task
    private int[] taskDuration;
    // Number of agents needed for each task at each time
    private int[][] agentsNeeded;
    // Daily disponibilities of each agent
    private int[][] dayDispo;
    // Hour disponibilities of each agent on working days
    private int[][] hourDispo;
    // The agent is available at time t
    private int[][] agentDispo;
    // The agent is able to perform the task i
    private int[][] agentSkills;
    // The agent is able to perform the task i and complete it
    // before the end of the day
    private int[][][] agentCanStart;

    // Hexaly Optimizer
    final HexalyOptimizer optimizer;
    // Decision variable
    private HxExpression[][][] agentStart;
    // Agent working time per day
    private HxExpression[][] agentWorkingTime;
    // Attributed number of agents for each task
    private HxExpression[][] attributedAgents;
    // Difference between needed number of agents and attributed number
    // of agents for each task
    private HxExpression[][] agentDiff;
    // Indicators for lack and excess of agents
    private HxExpression agentLack;
    private HxExpression agentExcess;
    // Difference between first and last hour of agents each day
    private HxExpression[][] agentDayStart;
    private HxExpression[][] agentDayEnd;
    private HxExpression agentDayLength;

    public WorkforceScheduling(HexalyOptimizer optimizer) throws IOException {
        this.optimizer = optimizer;
    }

    public void readInstance(String fileName) throws IOException {
        try (Scanner input = new Scanner(new File(fileName))) {
            input.useLocale(Locale.ROOT);
            nbAgents = input.nextInt();
            nbTasks = input.nextInt();
            nbDays = input.nextInt();
            horizon = 24 * nbDays;

            // Duration of each task
            input.nextLine();
            taskDuration = new int[nbTasks];
            for (int i = 0; i < nbTasks; ++i) {
                taskDuration[i] = input.nextInt();
            }

            // Number of agents needed for the task i at time t
            // agentsNeeded[i][t] contains the number of agents needed
            // for the task i at time t
            input.nextLine();
            input.nextLine();
            agentsNeeded = new int[nbTasks][horizon];
            for (int d = 0; d < nbDays; ++d) {
                for (int i = 0; i < nbTasks; ++i) {
                    for (int h = 0; h < 24; ++h) {
                        int t = d * 24 + h;
                        agentsNeeded[i][t] = input.nextInt();
                    }
                }
                input.nextLine();
            }

            // Agent disponibility
            // dayDispo[a][d] = 1 if agent a is available on day d
            dayDispo = new int[nbAgents][nbDays];
            for (int a = 0; a < nbAgents; ++a) {
                for (int d = 0; d < nbDays; ++d) {
                    dayDispo[a][d] = input.nextInt();
                }
            }

            // Hour disponibility
            input.nextLine();
            hourDispo = new int[nbAgents][24];
            startDay = new int[nbAgents];
            endDay = new int[nbAgents];
            for (int a = 0; a < nbAgents; ++a) {
                startDay[a] = input.nextInt();
                endDay[a] = input.nextInt();
                for (int h = 0; h < 24; ++h) {
                    if ((h >= startDay[a]) && (h < endDay[a])) {
                        hourDispo[a][h] = 1;
                    } else {
                        hourDispo[a][h] = 0;
                    }
                }
            }

            // We can concatenate these two informations
            // into a global indicator of agent disponibility
            // agentDispo[a][t] = 1 if agent a is available
            // on time t
            agentDispo = new int[nbAgents][horizon];
            for (int a = 0; a < nbAgents; ++a) {
                for (int d = 0; d < nbDays; ++d) {
                    for (int h = 0; h < 24; ++h) {
                        int t = 24 * d + h;
                        agentDispo[a][t] = dayDispo[a][d] * hourDispo[a][h];
                    }
                }
            }

            // Agent skills
            // agentSkills[a][i] = 1 if agent a is able to perform the task i
            input.nextLine();
            agentSkills = new int[nbAgents][nbTasks];
            for (int a = 0; a < nbAgents; ++a) {
                for (int i = 0; i < nbTasks; ++i) {
                    agentSkills[a][i] = input.nextInt();
                }
            }

            // We can use all these informations and task length to get an
            // indicator telling us if the agent a can begin the task i
            // at time t, and complete it before the end of day
            agentCanStart = new int[nbAgents][nbTasks][horizon];
            for (int a = 0; a < nbAgents; ++a) {
                for (int i = 0; i < nbTasks; ++i) {
                    for (int d = 0; d < nbDays; ++d) {
                        for (int h = 0; h < startDay[a]; ++h) {
                            int t = 24 * d + h;
                            agentCanStart[a][i][t] = 0;
                        }
                        int endPossible = endDay[a] - taskDuration[i] + 1;
                        for (int h = startDay[a]; h < endPossible; ++h) {
                            int t = 24 * d + h;
                            agentCanStart[a][i][t] = 
                                agentDispo[a][t] * agentSkills[a][i];
                        }
                        for (int h = endPossible; h < 24; ++h) {
                            int t = 24 * d + h;
                            agentCanStart[a][i][t] = 0;
                        }
                    }
                }
            }
        }
    }

    // Returns the first time step s such that
    // [s, s + duration) intersects time t
    public int intersectionStart(int t, int duration) {
        return Math.max(t - duration + 1, 0);
    }

    public void solve(int timeLimit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        // Definition of the decision variable :
        // agentStart[a][i][t] = 1 if agent a starts the task i at time t
        agentStart = new HxExpression[nbAgents][nbTasks][horizon];
        for (int a = 0; a < nbAgents; ++a) {
            for (int i = 0; i < nbTasks; ++i) {
                for (int t = 0; t < horizon; ++t) {
                    agentStart[a][i][t] = model.boolVar();
                }
            }
        }

        // An agent can only work if he is able to complete his task
        // before the end of the day
        for (int a = 0; a < nbAgents; ++a) {
            for (int i = 0; i < nbTasks; ++i) {
                for (int t = 0; t < horizon; ++t) {
                    model.constraint(
                        model.leq(agentStart[a][i][t], agentCanStart[a][i][t])
                    );
                }
            }
        }

        // Each agent can only work on one task at a time
        for (int a = 0; a < nbAgents; ++a) {
            for (int i1 = 0; i1 < nbTasks; ++i1) {
                for (int i2 = 0; i2 < nbTasks; ++i2) {
                    for (int t1 = 0; t1 < horizon; ++t1) {
                        int t2Start = intersectionStart(t1, taskDuration[i2]);
                        for (int t2 = t2Start; t2 < t1 + 1; ++t2) {
                            if ((i1 != i2) || (t1 != t2)) {
                                model.constraint(
                                    model.leq(
                                        model.sum(
                                            agentStart[a][i1][t1],
                                            agentStart[a][i2][t2]
                                        ),
                                        1
                                    )
                                );
                            }
                        }
                    }
                }
            }
        }

        // Working time per agent
        agentWorkingTime = new HxExpression[nbAgents][nbDays];
        for (int a = 0; a < nbAgents; ++a) {
            for (int d = 0; d < nbDays; ++d) {
                agentWorkingTime[a][d] = model.sum();
                for (int i = 0; i < nbTasks; ++i) {
                    for (int t = 24 * d; t < 24 * (d + 1); ++t) {
                        agentWorkingTime[a][d].addOperand(
                            model.prod(agentStart[a][i][t], taskDuration[i])
                        );
                    }
                }
            }
        }

        // Minimum and maximum amount of time worked
        for (int a = 0; a < nbAgents; ++a) {
            for (int d = 0; d < nbDays; ++d) {
                if (dayDispo[a][d] == 1) {
                    // 4 <= agentWorkingTime <= 8
                    model.constraint(
                        model.geq(agentWorkingTime[a][d], MIN_WORKING_TIME)
                    );
                    model.constraint(
                        model.leq(agentWorkingTime[a][d], MAX_WORKING_TIME)
                    );
                }
            }
        }

        // Difference between needed and attributed number
        // of agents for each task
        agentDiff = new HxExpression[nbTasks][horizon];
        attributedAgents = new HxExpression[nbTasks][horizon];
        for (int i = 0; i < nbTasks; ++i) {
            for (int t0 = 0; t0 < horizon; ++t0) {
                attributedAgents[i][t0] = model.sum();
                for (int a = 0; a < nbAgents; ++a) {
                    int tStart = intersectionStart(t0, taskDuration[i]);
                    for (int t = tStart; t < t0 + 1; ++t) {
                        attributedAgents[i][t0].addOperand(agentStart[a][i][t]);
                    }
                }
                agentDiff[i][t0] = 
                    model.sub(agentsNeeded[i][t0], attributedAgents[i][t0]);
            }
        }

        // Indicators for lack and excess of agents
        // Agent Lack
        agentLack = model.sum();
        for (int i = 0; i < nbTasks; ++i) {
            for (int t = 0; t < horizon; ++t) {
                HxExpression lack = model.max(agentDiff[i][t], 0);
                agentLack.addOperand(model.pow(lack, 2));
            }
        }

        // Agent Excess
        agentExcess = model.sum();
        for (int i = 0; i < nbTasks; ++i) {
            for (int t = 0; t < horizon; ++t) {
                HxExpression excess = model.max(
                    model.prod(-1, agentDiff[i][t]),
                    0
                );
                agentExcess.addOperand(model.pow(excess, 2));
            }
        }

        // Difference between first and last hour of agents each day
        agentDayLength = model.sum();
        for (int a = 0; a < nbAgents; ++a) {
            for (int d = 0; d < nbDays; ++d) {
                HxExpression endWork = model.max();
                HxExpression startWork = model.min();
                for (int i = 0; i < nbTasks; ++i) {
                    for (int t = 24 * d; t < 24 * (d + 1); ++t) {
                        endWork.addOperand(
                            model.prod(
                                (t + taskDuration[i]),
                                agentStart[a][i][t]
                            )
                        );
                        startWork.addOperand(
                            model.sum(
                                t,
                                model.prod(
                                    model.sub(1, agentStart[a][i][t]),
                                    horizon
                                )
                            )
                        );
                    }
                }
                HxExpression dayLength = model.max(
                    model.sub(endWork, startWork),
                    0
                );
                agentDayLength.addOperand(dayLength);
            }
        }

        // Objectives
        model.minimize(agentLack);
        model.minimize(agentExcess);
        model.minimize(agentDayLength);

        model.close();

        // Parameterize the optimizer
        optimizer.getParam().setTimeLimit(timeLimit);

        optimizer.solve();
    }

    public void writeSolution(String fileName) throws IOException {
        try (PrintWriter output = new PrintWriter(fileName)) {
            System.out.println("Solution written in file " + fileName);

            output.println(agentLack.getDoubleValue());
            output.println(agentExcess.getDoubleValue());
            output.println(agentDayLength.getValue());

            for (int a = 0; a < nbAgents; ++a) {
                for (int i = 0; i < nbTasks; ++i) {
                    for (int t = 0; t < horizon; ++t) {
                        output.print(agentStart[a][i][t].getValue() + " ");
                    }
                    output.println();
                }
                output.println();
            }
        }
    }

    public static void main(String[] args) {
        if (args.length < 1) {
            System.out.println("Usage: java WorkforceScheduling instanceFile " +
                    "[outputFile] [timeLimit]");
            System.exit(1);
        }

        String instanceFile = args[0];
        String outputFile = args.length > 1 ? args[1] : null;
        String strTimeLimit = args.length > 2 ? args[2] : "30";

        try (HexalyOptimizer optimizer = new HexalyOptimizer()) {
            WorkforceScheduling model = new WorkforceScheduling(optimizer);
            model.readInstance(instanceFile);
            model.solve(Integer.parseInt(strTimeLimit));
            if (outputFile != null) {
                model.writeSolution(outputFile);
            }
        } catch (Exception ex) {
            System.err.println(ex);
            ex.printStackTrace();
            System.exit(1);
        }
    }
}
