#include "optimizer/hexalyoptimizer.h"
#include <algorithm>
#include <fstream>
#include <iostream>
#include <limits>
#include <numeric>
#include <vector>

using namespace hexaly;

class WorkforceScheduling {
private:

    // Number of agents
    int nbAgents;
    // Number of tasks
    int nbTasks;
    // Start of the day for each agent
    std::vector<int> startDay;
    // End of the day for each agent
    std::vector<int> endDay;
    // Time horizon : number of days 
    int nbDays;
    // Time horizon : number of time steps
    int horizon;
    // Maximum and minimum worked hours per day
    int MIN_WORKING_TIME = 4;
    int MAX_WORKING_TIME = 8;
    // Duration of each task
    std::vector<int> taskDuration;
    // Number of agents needed for each task at each time
    std::vector<std::vector<int>> agentsNeeded;
    // Daily disponibilities of each agent
    std::vector<std::vector<int>> dayDispo;
    // Hour disponibilities of each agent on working days
    std::vector<std::vector<int>> hourDispo;
    // The agent is available at time t
    std::vector<std::vector<int>> agentDispo;
    // The agent is able to perform the task i
    std::vector<std::vector<int>> agentSkills;
    // The agent is able to perform the task i and complete it
    // before the end of the day
    std::vector<std::vector<std::vector<int>>> agentCanStart;

    // Hexaly Optimizer
    HexalyOptimizer optimizer;
    // Decision variable
    std::vector<std::vector<std::vector<HxExpression>>> agentStart;
    // Agent working time per day
    std::vector<std::vector<HxExpression>> agentWorkingTime;
    // Attributed number of agents for each task
    std::vector<std::vector<HxExpression>> attributedAgents;
    // Difference between needed number of agents and attributed number
    // of agents for each task
    std::vector<std::vector<HxExpression>> agentDiff;
    // Indicators for lack and excess of agents
    HxExpression agentLack;
    HxExpression agentExcess;
    // Difference between first and last hour of agents each day
    std::vector<std::vector<HxExpression>> agentDayStart;
    std::vector<std::vector<HxExpression>> agentDayEnd;
    HxExpression agentDayLength;

public:
    WorkforceScheduling(const std::string& fileName) : optimizer() {}

    void readInstance(const std::string& fileName) {
        std::ifstream inFile;
        inFile.exceptions(std::ifstream::failbit | std::ifstream::badbit);
        inFile.open(fileName.c_str());

        inFile >> nbAgents;
        inFile >> nbTasks;
        inFile >> nbDays;
        horizon = nbDays * 24;

        // The duration of each task
        taskDuration.resize(nbTasks);
        for (int i = 0; i < nbTasks; ++i) {
            inFile >> taskDuration[i];
        }

        // Number of agents needed for the task i at time t
        // agentsNeeded[i][t] contains the number of agents needed
        // for the task i at time t
        agentsNeeded.resize(nbTasks);
        for (int d = 0; d < nbDays; ++d) {
            for (int i = 0; i < nbTasks; ++i) {
                agentsNeeded[i].resize(horizon);
                for (int h = 0; h < 24; ++h) {
                    int t = d * 24 + h;
                    inFile >> agentsNeeded[i][t];
                }
            }
            
        }

        // Agent disponibility
        // dayDispo[a][d] = 1 if agent a is available on day d
        dayDispo.resize(nbAgents);
        for(int a = 0; a < nbAgents; ++a) {
            dayDispo[a].resize(nbDays);
            for(int d = 0; d < nbDays; ++d) {
                inFile >> dayDispo[a][d];
            }
        }

        // hourDispo[a][h] = 1 if agent a is available on hour h
        hourDispo.resize(nbAgents);
        startDay.resize(nbAgents);
        endDay.resize(nbAgents);
        for (int a = 0; a < nbAgents; ++a) {
            hourDispo[a].resize(24);
            inFile >> startDay[a];
            inFile >> endDay[a];
            for (int h = 0; h < 24; ++h) {
                if ((h >= startDay[a]) && (h < endDay[a])) {
                    hourDispo[a][h] = 1;
                } else {
                    hourDispo[a][h] = 0;
                }
            }
        }

        // We can concatenate these two informations
        // into a global indicator of agent disponibility
        // agentDispo[a][t] = 1 if agent a is available
        // on time t
        agentDispo.resize(nbAgents);
        for (int a = 0; a < nbAgents; ++a) {
            agentDispo[a].resize(horizon);
            for (int d = 0; d < nbDays; ++d) {
                for (int h = 0; h < 24; ++h) {
                    int t = 24 * d + h;
                    agentDispo[a][t] = dayDispo[a][d] * hourDispo[a][h];
                }
            }
        }

        // Agent skills
        // agentSkills[a][i] = 1 if agent a is able to perform the task i
        agentSkills.resize(nbAgents);
        for (int a = 0; a < nbAgents; ++a) {
            agentSkills[a].resize(nbTasks);
            for (int i = 0; i < nbTasks; ++i) {
                inFile >> agentSkills[a][i];
            }
        }

        // We can use all these informations and task length to get an indicator
        // telling us if the agent a can begin the task i at time t,
        // and complete it before the end of day
        agentCanStart.resize(nbAgents);
        for (int a = 0; a < nbAgents; ++a) {
            agentCanStart[a].resize(nbTasks);
            for (int i = 0; i < nbTasks; ++i) {
                agentCanStart[a][i].resize(horizon);
                for (int d = 0; d < nbDays; ++d) {
                    for (int h = 0; h < startDay[a]; ++h) {
                        int t = 24 * d + h;
                        agentCanStart[a][i][t] = 0;
                    }
                    int endPossible = endDay[a] - taskDuration[i] + 1;
                    for (int h = startDay[a]; h < endPossible; ++h) {
                        int t = 24 * d + h;
                        agentCanStart[a][i][t] =
                            agentDispo[a][t] * agentSkills[a][i];
                    }
                    for (int h = endPossible; h < 24; ++h) {
                        int t = 24 * d + h;
                        agentCanStart[a][i][t] = 0;
                    }
                }
            }
        }

        inFile.close();
    }

    // Returns the first time step s such that
    // [s, s + duration) intersects time t
    int intersectionStart(int t, int duration) {
        return std::max(t - duration + 1, 0);
    }

    void solve(int TimeLimit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        // Definition of the decision variable : 
        // agentStart[a][i][t] = 1 if agent a starts the task i at time t
        agentStart.resize(nbAgents);
        for (int a = 0; a < nbAgents; ++a) {
            agentStart[a].resize(nbTasks);
            for (int i = 0; i < nbTasks; ++i) {
                agentStart[a][i].resize(horizon);
                for (int t = 0; t < horizon; ++t) {
                    agentStart[a][i][t] = model.boolVar();
                }
            }
        }

        // An agent can only work if he is able to complete his task
        // before the end of the day
        for (int a = 0; a < nbAgents; ++a) {
            for (int i = 0; i < nbTasks; ++i) {
                for (int t = 0; t < horizon; ++t) {
                    model.constraint(
                        agentStart[a][i][t] <= agentCanStart[a][i][t]
                    );
                }
            }
        }
        
        // Each agent can only work on one task at a time
        for (int a = 0; a < nbAgents; ++a) {
            for (int i1 = 0; i1 < nbTasks; ++i1) {
                for (int i2 = 0; i2 < nbTasks; ++i2) {
                    for (int t1 = 0; t1 < horizon; ++t1) {
                        int t2Start = intersectionStart(t1, taskDuration[i2]);
                        for (int t2 = t2Start; t2 < t1 + 1; ++t2) {
                            if ((i1 != i2) || (t1 != t2)) {
                                model.constraint(
                                    agentStart[a][i1][t1] +
                                    agentStart[a][i2][t2] <= 1
                                );
                            }
                        }
                    }
                }
            }
        }

        // Working time per agent per day
        agentWorkingTime.resize(nbAgents);
        for (int a = 0; a < nbAgents; ++a) {
            agentWorkingTime[a].resize(nbDays);
            for (int d = 0; d < nbDays; ++d) {
                agentWorkingTime[a][d] = model.sum();
                for (int i = 0; i < nbTasks; ++i) {
                    for (int t = 24 * d; t < 24 * (d + 1); ++t) {
                        agentWorkingTime[a][d].addOperand(
                            agentStart[a][i][t] * taskDuration[i]
                        );
                    }
                }
            }
        }

        // Minimum and maximum amount of time worked
        for (int a = 0; a < nbAgents; ++a) {
            for (int d = 0; d < nbDays; ++d) {
                if (dayDispo[a][d] == 1) {
                    model.constraint(
                        agentWorkingTime[a][d] >= MIN_WORKING_TIME
                    );
                    model.constraint(
                        agentWorkingTime[a][d] <= MAX_WORKING_TIME
                    );
                }
            }
        } 

        // Difference between needed and attributed number 
        // of agents for each task
        agentDiff.resize(nbTasks);
        attributedAgents.resize(nbTasks);
        for (int i = 0; i < nbTasks; ++i) {
            agentDiff[i].resize(horizon);
            attributedAgents[i].resize(horizon);
            for (int t0 = 0; t0 < horizon; ++t0) {
                attributedAgents[i][t0] = model.sum();
                for (int a = 0; a < nbAgents; ++a) {
                    int tStart = intersectionStart(t0, taskDuration[i]);
                    for (int t = tStart; t < t0 + 1; ++t) {
                        attributedAgents[i][t0].addOperand(agentStart[a][i][t]);
                    }
                }
                agentDiff[i][t0] = 
                        agentsNeeded[i][t0] - attributedAgents[i][t0];
            }
        }

        // Indicators for lack and excess of agents
        // Agent Lack
        agentLack = model.sum();
        for (int i = 0; i < nbTasks; ++i) {
            for (int t = 0; t < horizon; ++t) {
                HxExpression lack = model.max(agentDiff[i][t], 0);
                agentLack.addOperand(model.pow(lack, 2));
            }
        }

        // Agent Excess
        agentExcess = model.sum();
        for (int i = 0; i < nbTasks; ++i) {
            for (int t = 0; t < horizon; ++t) {
                HxExpression excess = model.max(-1 * agentDiff[i][t], 0);
                agentExcess.addOperand(model.pow(excess, 2));
            }
        }

        // Difference between first and last hour of agents each day
        agentDayLength = model.sum();
        for (int a = 0; a < nbAgents; ++a) {
            for (int d = 0; d < nbDays; ++d) {
                HxExpression endWork = model.max();
                HxExpression startWork = model.min();
                for (int i = 0; i < nbTasks; ++i) {
                    for (int t = 24 * d; t < 24 * (d + 1); ++t) {
                        endWork.addOperand(
                            (t + taskDuration[i]) * agentStart[a][i][t]
                        );
                        startWork.addOperand(
                            t + (1 - agentStart[a][i][t]) * horizon
                        );
                    }
                }

                HxExpression dayLength = model.max(endWork - startWork, 0);
                agentDayLength.addOperand(dayLength);
            }
        }

        // Objectives
        model.minimize(agentLack);
        model.minimize(agentExcess);
        model.minimize(agentDayLength);

        model.close();

        // Parameterize the optimizer
        optimizer.getParam().setTimeLimit(TimeLimit);

        optimizer.solve();
    }

    /* Write the solution in a file */
    void writeSolution(const std::string& fileName) {
        std::ofstream outfile(fileName.c_str());
        if (!outfile.is_open()) {
            std::cerr << "File " << fileName << " cannot be opened." 
                << std::endl;
            exit(1);
        }
        std::cout << "Solution written in file " << fileName << std::endl;

        outfile << agentLack.getDoubleValue() << std::endl;
        outfile << agentExcess.getDoubleValue() << std::endl;
        outfile << agentDayLength.getValue() << std::endl;
        for (int a = 0; a < nbAgents; ++a) {
            for (int i = 0; i < nbTasks; ++i) {
                for (int t = 0; t < horizon; ++t) {
                    outfile << agentStart[a][i][t].getValue() << " ";
                }
                outfile << std::endl;
            }
            outfile << std::endl;
        }
        outfile.close();
    }
};

int main(int argc, char** argv) {
    if (argc < 2) {
        std::cout << "Usage: workforce_scheduling instanceFile "
            << "[outputFile] [timeLimit]" << std::endl;
        exit(1);
    }

    const char* instanceFile = argv[1];
    const char* outputFile = argc > 2 ? argv[2] : NULL;
    const char* strTimeLimit = argc > 3 ? argv[3] : "30";

    WorkforceScheduling model(instanceFile);
    try {
        model.readInstance(instanceFile);
        const int timeLimit = atoi(strTimeLimit);
        model.solve(timeLimit);
        if (outputFile != NULL)
            model.writeSolution(outputFile);
        return 0;
    } catch (const std::exception& e) {
        std::cerr << "An error occurred: " << e.what() << std::endl;
        return 1;
    }
}