import hexaly.optimizer
import sys


# Function used to extract the information from the data file
def readData(filename):
    with open(filename) as f:
        lines = iter(f.readlines())

    # Dimensions of the problem
    nb_agents = int(next(lines))
    nb_tasks = int(next(lines))
    nb_days = int(next(lines))
    horizon = nb_days * 24
    min_working_time = 4
    max_working_time = 8

    # Tasks duration
    next(lines)
    task_duration = [0 for i in range(nb_tasks)]
    for i in range(nb_tasks):
        task_duration[i] = int(next(lines))

    # Number of agents needed for the task i at time t
    # agents_needed[i][t] contains the number of agents needed
    # for the task i at time t
    next(lines)
    next(lines)
    agents_needed = [[0 for t in range(horizon)] for i in range(nb_tasks)]
    for d in range(nb_days):
        for i in range(nb_tasks):
            elements = next(lines).split(" ")
            for h in range(24):
                t = d * 24 + h
                agents_needed[i][t] = int(elements[h])
        next(lines)

    # Agent disponibility
    # day_dispo[a][d] = 1 if agent a is available on day d
    day_dispo = [[0 for d in range(nb_days)] for a in range(nb_agents)]
    for a in range(nb_agents):
        elements = next(lines).split(" ")
        for d in range(nb_days):
            day_dispo[a][d] = int(elements[d])

    # hour_dispo[a][h] = 1 if agent a is available on hour h
    next(lines)
    hour_dispo = [[0 for h in range(24)] for a in range(nb_agents)]
    start_day = [0 for a in range(nb_agents)]
    end_day = [0 for a in range(nb_agents)]
    for a in range(nb_agents):
        elements = next(lines).split(" ")

        start_day[a] = int(elements[0])
        end_day[a] = int(elements[1])

        hour_dispo[a][0:start_day[a]] = [0] * start_day[a]
        hour_dispo[a][start_day[a]:end_day[a]] = \
            [1] * (end_day[a] - start_day[a])
        hour_dispo[a][end_day[a]:24] = [0] * (24 - end_day[a])

    # We can concatenate these two informations
    # into a global indicator of agent disponibility
    # agent_dispo[a][t] = 1 if agent a is available
    # on time t
    agent_dispo = [[0 for t in range(horizon)] for a in range(nb_agents)]
    for a in range(nb_agents):
        for d in range(nb_days):
            for h in range(24):
                t = d * 24 + h
                agent_dispo[a][t] = day_dispo[a][d] * hour_dispo[a][h]

    # Agent skills
    # agent_skills[a][i] = 1 if agent a is able to perform the task i
    next(lines)
    agent_skills = [[0 for i in range(nb_tasks)] for a in range(nb_agents)]
    for a in range(nb_agents):
        elements = next(lines).split(" ")

        for i in range(nb_tasks):
            agent_skills[a][i] = int(elements[i])

    # We can use all these informations and task length to get an indicator
    # telling us if the agent a can begin the task i at time t,
    # and complete it before the end of day
    agent_can_start = [
        [
            [0 for t in range(horizon)]
            for i in range(nb_tasks)
        ]
        for a in range(nb_agents)
    ]
    for a in range(nb_agents):
        for i in range(nb_tasks):
            for d in range(nb_days):
                for h in range(0, start_day[a]):
                    t = d * 24 + h
                    agent_can_start[a][i][t] = 0
                for h in range(start_day[a], end_day[a] - task_duration[i] + 1):
                    t = d * 24 + h
                    agent_can_start[a][i][t] = agent_dispo[a][t] * \
                        agent_skills[a][i]
                for h in range(end_day[a] - task_duration[i] + 1, 24):
                    t = d * 24 + h
                    agent_can_start[a][i][t] = 0

    return nb_agents, nb_tasks, nb_days, horizon, task_duration, agents_needed, \
        day_dispo, agent_can_start, min_working_time, max_working_time


# Returns the time steps corresponding to day d
def times_of_day(d):
    return range(d * 24, (d + 1) * 24)


# Returns the time steps s such that
# [s, s + duration) intersects time t
def intersecting_starts(t, duration):
    return range(max(t - duration + 1, 0), t + 1)


def main(instance_file, output_file, time_limit):
    nb_agents, nb_tasks, nb_days, horizon, task_duration, \
        agents_needed, day_dispo, agent_can_start, \
        min_working_time, max_working_time = readData(instance_file)

    with hexaly.optimizer.HexalyOptimizer() as optimizer:
        #
        # Declare the optimization model
        #
        model = optimizer.model

        # Definition of the decision variable :
        # agent_start[a][i][t] = 1 if agent a starts the task i at time t
        agent_start = [
            [
                [model.bool() for t in range(horizon)]
                for i in range(nb_tasks)
            ]
            for a in range(nb_agents)
        ]

        # An agent can only work if he is able to complete his task
        # before the end of the day
        for a in range(nb_agents):
            for i in range(nb_tasks):
                for t in range(horizon):
                    model.constraint(
                        agent_start[a][i][t] <= agent_can_start[a][i][t]
                    )

        # Each agent can only work on one task at a time
        for a in range(nb_agents):
            for i1 in range(nb_tasks):
                for i2 in range(nb_tasks):
                    for t1 in range(horizon):
                        for t2 in intersecting_starts(t1, task_duration[i2]):
                            if (i1 != i2) or (t1 != t2):
                                model.constraint(
                                    agent_start[a][i1][t1] +
                                    agent_start[a][i2][t2] <= 1
                                )

        # Working time per agent per day
        agent_working_time = [
            [
                sum(
                    agent_start[a][i][t] * task_duration[i]
                    for i in range(nb_tasks)
                    for t in times_of_day(d)
                )
                for d in range(nb_days)
            ]
            for a in range(nb_agents)
        ]

        # Minimum and maximum amount of time worked
        for a in range(nb_agents):
            for d in range(nb_days):
                if day_dispo[a][d] == 1:
                    model.constraint(
                        agent_working_time[a][d] >= min_working_time
                    )
                    model.constraint(
                        agent_working_time[a][d] <= max_working_time
                    )

        # Difference between needed and attributed number of agents
        # for each task
        agent_attributed = [
            [
                sum(
                    agent_start[a][i][t]
                    for a in range(nb_agents)
                    for t in intersecting_starts(t0, task_duration[i])
                )
                for t0 in range(horizon)
            ]
            for i in range(nb_tasks)
        ]

        agent_diff = [
            [
                agents_needed[i][t] - agent_attributed[i][t]
                for t in range(horizon)
            ]
            for i in range(nb_tasks)
        ]

        # Indicators for lack and excess of agents
        agent_lack = sum(
            model.pow(model.max(agent_diff[i][t], 0), 2)
            for t in range(horizon)
            for i in range(nb_tasks)
        )

        agent_excess = sum(
            model.pow(model.max(-agent_diff[i][t], 0), 2)
            for t in range(horizon)
            for i in range(nb_tasks)
        )

        agent_day_start = [
            [
                model.min(
                    t + (1 - agent_start[a][i][t]) * horizon
                    for i in range(nb_tasks)
                    for t in times_of_day(d)
                )
                for d in range(nb_days)
            ]
            for a in range(nb_agents)
        ]

        agent_day_end = [
            [
                model.max(
                    (t + task_duration[i]) * agent_start[a][i][t]
                    for i in range(nb_tasks)
                    for t in times_of_day(d)
                )
                for d in range(nb_days)
            ]
            for a in range(nb_agents)
        ]

        # Difference between first and last hour of agents each day
        agent_day_length = sum(
            model.max(agent_day_end[a][d] - agent_day_start[a][d], 0)
            for a in range(nb_agents)
            for d in range(nb_days)
        )

        # Objectives
        model.minimize(agent_lack)
        model.minimize(agent_excess)
        model.minimize(agent_day_length)

        model.close()

        # Parameterize the optimizer
        optimizer.param.time_limit = time_limit

        optimizer.solve()
        print("I solved the model")
        # Outputs if output file specified
        if output_file != None:
            with open(output_file, "w") as f:
                print("Solution written in file", output_file)
                # Objective values
                f.write(str(agent_lack.value) + "\n")
                f.write(str(agent_excess.value) + "\n")
                f.write(str(agent_day_length.value) + "\n")

                # Decision variable
                for a in range(nb_agents):
                    for i in range(nb_tasks):
                        for t in range(horizon):
                            f.write(str(agent_start[a][i][t].value) + " ")
                        f.write("\n")
                    f.write("\n")


# Main
if __name__ == '__main__':
    if len(sys.argv) < 2:
        print("Usage: python workforce_scheduling.py instance_file "
              + "[output_file] [time_limit]")
        sys.exit(1)

    instance_file = sys.argv[1]
    output_file = sys.argv[2] if len(sys.argv) >= 3 else None
    time_limit = int(sys.argv[3]) if len(sys.argv) >= 4 else 30
    main(instance_file, output_file, time_limit)
