using System;
using System.IO;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Globalization;
using Hexaly.Optimizer;

/**
 * Activity class represents a work item with its scheduling constraints
 * - id: Unique identifier for the activity
 * - minStart: minimum possible start time (in seconds)
 * - maxEnd: maximum possible end time (in seconds)
 * - duration: Duration of the activity (in seconds)
 */
class Activity
{
    public int activityId;
    public int actMinStart;
    public int actMaxEnd;
    public int actDuration;

    public Activity(int actId, int actMinStart, int actMaxEnd, int actDuration)
    {
        this.activityId = actId;
        this.actMinStart = actMinStart;
        this.actMaxEnd = actMaxEnd;
        this.actDuration = actDuration;
    }
}

/**
 * Agent class represents an employee with their availability window
 * - id: Unique identifier for the agent
 * - availabilityStart: Start of availability period (in seconds)
 * - availabilityEnd: End of availability period (in seconds)
 */
class Agent
{
    public int agentId;
    public int availabilityStart;
    public int availabilityEnd;

    public Agent(int agentId, int avStart, int avEnd)
    {
        this.agentId = agentId;
        this.availabilityStart = avStart;
        this.availabilityEnd = avEnd;
    }
}

/**
 * Shift class represents a scheduled work period for an activity
 * - id: Unique identifier for the shift
 * - activityId: ID of the activity performed
 * - shiftStart: Start time of the shift (in seconds)
 * - shiftEnd: End time of the shift (in seconds)
 */
public class Shift
{
    public int shiftId;
    public int shiftActivity;
    public int shiftStart;
    public int shiftEnd;

    public Shift(int id, int actId, int start, int end)
    {
        this.shiftId = id;
        this.shiftActivity = actId;
        this.shiftStart = start;
        this.shiftEnd = end;
    }
}

public class WorkforceSchedulingShifts : IDisposable
{
    // Dimensions of the problem
    int nbAgents;
    int nbActivities;
    int timeHorizon;
    int shiftIncrement;

    // Activities
    Activity[] activities;

    // Agents
    Agent[] agents;

    // Shifts
    int nbShifts;
    Shift[] shifts;

    // Hexaly Optimizer
    HexalyOptimizer optimizer;

    // Decision variables
    HxExpression[,] agentShift;

    // Objectives
    HxExpression totalUnderstaffing;
    HxExpression totalWorkingTime;

    public WorkforceSchedulingShifts()
    {
        optimizer = new HexalyOptimizer();
    }
    public void Dispose()
    {
        if (optimizer != null)
            optimizer.Dispose();
    }

    private void Solve(int limit)
    {

        // Declare the optimization model
        HxModel model = optimizer.GetModel();
        // Decision variables : agentShift[a, s] is true when the agent a works the shift s
        agentShift = new HxExpression[nbAgents, nbShifts];
        for (int i = 0; i < nbAgents; i++)
        {
            for (int j = 0; j < nbShifts; j++)
            {
                agentShift[i, j] = model.Bool();
            }
        }

        // Constraints
        // 1. The shifts performed by an agent must not overlap
        for (int a = 0; a < nbAgents; a++)
        {
            for (int s1 = 0; s1 < nbShifts; s1++)
            {
                for (int s2 = 0; s2 < nbShifts; s2++)
                {
                    if (OverlappingShifts(shifts[s1], shifts[s2]) && s1 != s2)
                    {
                        model.Constraint(model.Leq(model.Sum(agentShift[a, s1], agentShift[a, s2]), 1));
                    }
                }
            }
        }

        // 2. An agent must take at least a one-hour break between two shifts
        for (int a = 0; a < nbAgents; a++)
        {
            for (int s1 = 0; s1 < nbShifts; s1++)
            {
                for (int s2 = 0; s2 < nbShifts; s2++)
                {
                    if (NotEnoughBreak(shifts[s1], shifts[s2]) && s1 != s2)
                    {
                        model.Constraint(model.Leq(model.Sum(agentShift[a, s1], agentShift[a, s2]), 1));
                    }
                }
            }
        }

        // 3. Each agent must work exactly two shifts in the day
        for (int a = 0; a < nbAgents; a++)
        {
            HxExpression shiftsDone = model.Sum();
            for (int id = 0; id < nbShifts; id++)
            {
                shiftsDone.AddOperand(agentShift[a, id]);
            }
            model.Constraint(model.Eq(shiftsDone, 2));
        }

        // Objective 1 : Minimize understaffing
        HxExpression totalUnderstaffing = model.Sum();
        for (int i = 0; i < nbActivities; i++)
        {
            int currentTime = activities[i].actMinStart;
            while (currentTime < activities[i].actMaxEnd)
            {
                // Current interval of time
                int currentIntervalStart = currentTime;
                int currentIntervalEnd = currentTime + shiftIncrement;
                // The activity will not be pursued after its maximum possible end 
                if (currentIntervalEnd > activities[i].actMaxEnd)
                {
                    currentIntervalEnd = activities[i].actMaxEnd;
                }
                // Shifts performed in the current time interval 
                HxExpression totalAgentsWorking = model.Sum();
                for (int j = 0; j < nbShifts; j++)
                {
                    if (shifts[j].shiftActivity != i)
                    {
                        continue;
                    }
                    if (OverlappingIntervals(currentIntervalStart, currentIntervalEnd, shifts[j].shiftStart, shifts[j].shiftEnd))
                    {
                        for (int a = 0; a < nbAgents; a++)
                        {
                            totalAgentsWorking.AddOperand(agentShift[a, j]);
                        }
                    }
                }
                // Number of agents missing
                totalUnderstaffing.AddOperand(model.Prod(shiftIncrement, model.Max(0, model.Sub(1, totalAgentsWorking))));
                currentTime += shiftIncrement;
            }
        }
        model.Minimize(totalUnderstaffing);

        // Objective 2 : Minimize the total worktime of all the agents
        HxExpression totalWorkingTime = model.Sum();
        for (int a = 0; a < nbAgents; a++)
        {
            for (int s = 0; s < nbShifts; s++)
            {
                totalWorkingTime.AddOperand(model.Prod(agentShift[a, s], activities[shifts[s].shiftActivity].actDuration));
            }
        }
        model.Minimize(totalWorkingTime);
        model.Close();

        // Parametrize the optimizer
        optimizer.GetParam().SetTimeLimit(10);
        optimizer.Solve();
    }

    /**
    * Reads and parses input data from file
    * @param fileName Path to input file
    */
    void ReadInstance(string fileName)
    {
        using (StreamReader input = new StreamReader(fileName))
        {
            string[] splitted;
            // Read problem dimensions
            nbAgents = int.Parse(ReadLineIgnoringComments(input));
            nbActivities = int.Parse(ReadLineIgnoringComments(input));
            timeHorizon = int.Parse(ReadLineIgnoringComments(input));
            shiftIncrement = (int)(double.Parse(ReadLineIgnoringComments(input), CultureInfo.InvariantCulture) * 3600);

            // Read activities informations
            // Activity duration
            activities = new Activity[nbActivities];
            for (int i = 0; i < nbActivities; i++)
            {
                activities[i] = new Activity(0, 0, 0, int.Parse(ReadLineIgnoringComments(input)) * 3600);
            }
            // Possible period to do the activity
            for (int i = 0; i < nbActivities; i++)
            {
                splitted = ReadLineIgnoringComments(input).Split(new[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
                activities[i].actMinStart = int.Parse(splitted[0]) * 3600;
                activities[i].actMaxEnd = int.Parse(splitted[1]) * 3600;
            }

            //Read agents information
            agents = new Agent[nbAgents];
            // Availability period in the day of each agent
            for (int i = 0; i < nbAgents; i++)
            {
                splitted = ReadLineIgnoringComments(input).Split(new[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
                agents[i] = new Agent(i, int.Parse(splitted[0]) * 3600, int.Parse(splitted[1]) * 3600);
            };

            // Generation of all possible shifts for each activity with a given increment
            shifts = GenerateShifts(activities, shiftIncrement, shifts);

            // Validates the instance data
            ValidateInstance();
        }
    }

    /**
     * Validates the instance data for consistency
     */
    public void ValidateInstance()
    {
        // Check if all activities can be completed within their time windows
        foreach (Activity activity in activities)
        {
            if (activity.actDuration > (activity.actMaxEnd - activity.actMinStart))
            {
                throw new ArgumentException($"Activity {activity.activityId} duration ({activity.actDuration / 3600}h) exceeds its time window");
            }
        }

        // Check if workers' availability windows are valid
        foreach (Agent agent in agents)
        {
            if (agent.availabilityStart >= agent.availabilityEnd)
            {
                throw new ArgumentException($"Agent {agent.agentId} has invalid availability window");
            }
        }

        // Check if activities can be performed within workers' availability
        int earliestStart = activities.Min(a => a.actMinStart);
        int latestEnd = activities.Max(a => a.actMaxEnd);

        foreach (Agent agent in agents)
        {
            if (agent.availabilityStart > earliestStart || agent.availabilityEnd < latestEnd)
            {
                Console.WriteLine($"Warning: Agent {agent.agentId} availability might not cover all tasks");
            }
        }
    }

    /**
     * Ignore comments in a text file
     * @param input scanner object, reading data
     * @return next line in the file, ignoring comments
     */
    string ReadLineIgnoringComments(StreamReader input)
    {
        string line;
        do
        {
            line = input.ReadLine();
            if (line == null) return null;
            line = line.Split('#')[0].Trim();
        } while (string.IsNullOrEmpty(line));
        return line;
    }

    /**
    * Generates all possible shifts for activities with given time increment
    * @param activities Array of Activity objects
    * @param shiftIncrement Time increment in seconds
    * @return Array of generated Shift objects
    */
    Shift[] GenerateShifts(Activity[] activities, int shiftIncrement, Shift[] shifts)
    {
        int i = 0;
        for (int j = 0; j < nbActivities; j++)
        {
            int currentTime = activities[j].actMinStart;
            while (currentTime + activities[j].actDuration <= activities[j].actMaxEnd)
            {
                Array.Resize(ref shifts, i + 1);
                shifts[i] = new Shift(i, j, currentTime, currentTime + activities[j].actDuration);
                currentTime += shiftIncrement;
                i += 1;
            }
        }
        nbShifts = i;
        return shifts;
    }

    /**
    * Checks if there is at least a one-hour break between two shifts
    * @param s1 First shift
    * @param s2 Second shift
    * @return true if there is not enough break, false otherwise
    */
    bool NotEnoughBreak(Shift s1, Shift s2)
    {
        return !(s1.shiftStart >= s2.shiftEnd + 3600 || s1.shiftEnd + 3600 <= s2.shiftStart);
    }

    /**
    * Checks if two shifts are compatible
    * @param s1 First shift
    * @param s2 Second shift
    * @return true if shifts overlap, false otherwise
    */
    bool OverlappingShifts(Shift s1, Shift s2)
    {
        return !(s1.shiftStart >= s2.shiftEnd || s1.shiftEnd <= s2.shiftStart);
    }

    /**
    * Checks if two time intervals overlap
    * @param interval1Start Start of first interval
    * @param interval1End End of first interval
    * @param interval2Start Start of second interval
    * @param interval2End End of second interval
    * @return true if intervals overlap, false otherwise
    */
    bool OverlappingIntervals(int interval1Start, int interval1End, int interval2Start, int interval2End)
    {
        return !(interval1Start >= interval2End || interval1End <= interval2Start);
    }

    public static void Main(string[] args)
    {
        if (args.Length < 1)
        {
            Console.WriteLine("Usage: WorkforceSchedulingShifts inputFile [solFile] [timeLimit]");
            Environment.Exit(1);
        }
        string instanceFile = args[0];
        string outputFile = args.Length > 1 ? args[1] : null;
        string strTimeLimit = args.Length > 2 ? args[2] : "20";

        using (WorkforceSchedulingShifts model = new WorkforceSchedulingShifts())
        {
            model.ReadInstance(instanceFile);
            model.Solve(int.Parse(strTimeLimit));
        }
    }
}