import java.util.*;
import java.io.*;

import com.hexaly.optimizer.*;

/**
 * Activity class represents a work item with its scheduling constraints
 * - id: Unique identifier for the activity
 * - minStart: minimum possible start time (in seconds)
 * - maxEnd: maximum possible end time (in seconds)
 * - duration: Duration of the activity (in seconds)
 */
class Activity{
    int activityId;
    int actMinStart;
    int actMaxEnd;
    int actDuration;

    Activity(int actId, int actMinStart, int actMaxEnd, int actDuration){
        this.activityId = actId;
        this.actMinStart = actMinStart;
        this.actMaxEnd = actMaxEnd;
        this.actDuration = actDuration;
    }
}

/**
 * Agent class represents an employee with their availability window
 * - id: Unique identifier for the agent
 * - availabilityStart: Start of availability period (in seconds)
 * - availabilityEnd: End of availability period (in seconds)
 */
class Agent{
    int agentId;
    int availabilityStart;
    int availabilityEnd;

    Agent(int agentId, int avStart, int avEnd){
        this.agentId = agentId;
        this.availabilityStart = avStart;
        this.availabilityEnd = avEnd;
    }
} 

/**
 * Shift class represents a scheduled work period for an activity
 * - id: Unique identifier for the shift
 * - activityId: ID of the activity performed
 * - shiftStart: Start time of the shift (in seconds)
 * - shiftEnd: End time of the shift (in seconds)
 */
class Shift{
    int shiftId;
    int shiftActivity;
    int shiftStart;
    int shiftEnd;

    Shift(int shiftId, int shiftActivity, int shiftStart,  int shiftEnd){
        this.shiftId = shiftId;
        this.shiftActivity = shiftActivity;
        this.shiftStart = shiftStart;
        this.shiftEnd = shiftEnd;
    }
}

public class WorkforceSchedulingShifts {
    private final HexalyOptimizer optimizer;
    // Dimensions of the problem
    int nbAgents;
    int nbActivities;
    int timeHorizon;
    int shiftIncrement;

    // Activities
    Activity[] activities;

    // Agents
    Agent[] agents;

    // Shifts
    int nbShifts;
    Shift[] shifts;

    // Decision variables: time range of each task
    HxExpression[][] agentShift;
    // Objective = minimize the missing agent time and total working time of the agents
    HxExpression totalUnderstaffing;
    HxExpression totalWorkingTime;

    private WorkforceSchedulingShifts(HexalyOptimizer optimizer) {
        this.optimizer = optimizer;
    }

    private void solve(int limit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();
        agentShift = new HxExpression[nbAgents][nbShifts];
        for(int i = 0; i< nbAgents; i++){
            for(int j = 0 ; j < nbShifts; j++){
                agentShift[i][j] = model.boolVar();
            }
        }
        // Constraints
        // 1. The shifts performed by an agent must not overlap
        for (int a = 0; a < nbAgents; a++){
            for(int s1 = 0 ; s1 < nbShifts; s1++){
                for(int s2 = 0; s2 < nbShifts; s2++){
                    if(overlappingShifts(shifts[s1], shifts[s2]) && s1 != s2){
                        model.constraint(model.leq(model.sum(agentShift[a][s1], agentShift[a][s2]), 1));
                    }
                }
            }
        }
        // 2. An agent must take at least a one-hour break between two shifts
        for (int a=0; a < nbAgents; a++){
            for(int s1 = 0; s1 < nbShifts; s1++){
                for(int s2 = 0; s2 < nbShifts; s2++){
                    if(notEnoughBreak(shifts[s1], shifts[s2]) && s1 != s2){
                        model.constraint(model.leq(model.sum(agentShift[a][s1], agentShift[a][s2]), 1));
                    }
                }
            }
        }
        // 3. Each agent must work exactly two shifts in the day
        for(int a = 0; a < nbAgents; a++){
            HxExpression shiftsDone = model.sum();
            for(int id = 0; id < nbShifts; id++){
                shiftsDone.addOperand(agentShift[a][id]);
            }
            model.constraint(model.eq(shiftsDone, 2));
        }

        // Objective 1 : Minimize understaffing
        HxExpression totalUnderstaffing=model.sum();
        for(int i = 0; i < nbActivities; i++){
            int currentTime = activities[i].actMinStart;
            while (currentTime < activities[i].actMaxEnd){
                // Current interval of time
                int currentIntervalStart = currentTime;
                int currentIntervalEnd = currentTime + shiftIncrement;
                // The activity will not be pursued after its maximum possible end 
                if (currentIntervalEnd > activities[i].actMaxEnd){
                    currentIntervalEnd = activities[i].actMaxEnd;
                } 
                // Shifts performed in the current time interval 
                HxExpression totalAgentsWorking=model.sum();
                for(int j = 0 ; j < nbShifts; j++){
                    if(shifts[j].shiftActivity != i){
                        continue;
                    }
                    if(overlappingIntervals(currentIntervalStart, currentIntervalEnd, shifts[j].shiftStart, shifts[j].shiftEnd)){
                        for(int a = 0; a < nbAgents; a++){
                            totalAgentsWorking.addOperand(agentShift[a][j]);
                        }
                    }
                }
                // Number of agents missing
                totalUnderstaffing.addOperand(model.prod(shiftIncrement, model.max(0, model.sub(1, totalAgentsWorking))));
                currentTime += shiftIncrement;
            }
        }
        model.minimize(totalUnderstaffing);
        // Objective 2 : Minimize the total worktime of all the agents
        HxExpression totalWorkingTime = model.sum();
        for (int a = 0; a < nbAgents; a++){
            for(int s = 0; s < nbShifts; s++){
                totalWorkingTime.addOperand(model.prod(agentShift[a][s], activities[shifts[s].shiftActivity].actDuration));
            }
        }
        model.minimize(totalWorkingTime);
        model.close();
        // Parametrize the optimizer
        optimizer.getParam().setTimeLimit(10);
        optimizer.solve();
    }

    /**
     * Ignore comments in a text file
     * @param input scanner object, reading data
     * @return next int in the file, ignoring comments
     */
    private static int nextIntIgnoringComments(Scanner input) {
        while (input.hasNextLine()) {
            String line = input.nextLine().split("#")[0].trim(); 
            if (!line.isEmpty()) {
            Scanner lineScanner = new Scanner(line);
            if (lineScanner.hasNextInt()) {
                return lineScanner.nextInt();
            }
        }
        }
        throw new IllegalArgumentException("No valid integer found");
    }

    /**
     * Ignore comments in a text file
     * @param input scanner object, reading data
     * @return next double in the file, ignoring comments
     */
    private static double nextDoubleIgnoringComments(Scanner input) {
        while (input.hasNextLine()) {
            String line = input.nextLine().split("#")[0].trim();
            if (!line.isEmpty()) {
                Scanner lineScanner = new Scanner(line).useLocale(Locale.ROOT);
                if (lineScanner.hasNextDouble()) {
                    return lineScanner.nextDouble();
                }
            }
        }
        throw new IllegalArgumentException("No valid double found");
    }

    /**
     * Ignore comments in a text file
     * @param input scanner object, reading data
     * @return next line in the file, ignoring comments
     */
    private static String nextLineIgnoringComments(Scanner input) {
        while (input.hasNextLine()) {
            String line = input.nextLine().split("#")[0].trim();
            if (!line.isEmpty()) {
                return line;
            }
        }
        throw new IllegalArgumentException("No valid line found");
    }

    /* 
    * Read instance data
    */
    private void readInstance(String fileName) throws IOException {
        try (Scanner input = new Scanner(new File(fileName))) {
            // Read problem dimensions
            input.useLocale(Locale.ROOT);
            nbAgents = nextIntIgnoringComments(input);
            nbActivities = nextIntIgnoringComments(input);
            timeHorizon = nextIntIgnoringComments(input);
            shiftIncrement = (int) (nextDoubleIgnoringComments(input) * 3600);

            // Read activities informations
            activities = new Activity[nbActivities]; 
            for(int i = 0; i < nbActivities; i++){
                activities[i]= new Activity(i, 0, 0, 0);
                activities[i].actDuration = nextIntIgnoringComments(input) * 3600;
            }
            for(int i = 0; i < nbActivities; i++){
                String line = nextLineIgnoringComments(input);
                Scanner lineScanner = new Scanner(line);
                activities[i].actMinStart = lineScanner.nextInt() * 3600;
                activities[i].actMaxEnd = lineScanner.nextInt() * 3600;
            }

            // Read agents informations
            agents = new Agent[nbAgents];
            for(int i = 0; i < nbAgents; i++){
                String line = nextLineIgnoringComments(input);
                Scanner lineScanner = new Scanner(line);
                agents[i] = new Agent(i, 0, 0);
                agents[i].availabilityStart = lineScanner.nextInt() * 3600;
                agents[i].availabilityEnd = lineScanner.nextInt() * 3600;
            }
            // Generation of all possible shifts for each activity with a given increment
            generateShifts(activities, shiftIncrement); 

            // Validates the instance data
            validateInstance();
        }
    }

    /**
     * Generates all possible shifts for activities with given time increment
     * @param activities Array of Activity objects
     * @param shiftIncrement Time increment in seconds
     * @return Array of generated Shift objects
     */
    void generateShifts(Activity[] activities, int shiftIncrement){
        int i = 0;
        shifts = new Shift[]{};
        for(int j =0; j< nbActivities ; j++){
            int currentTime = activities[j].actMinStart;
            while(currentTime+activities[j].actDuration <= activities[j].actMaxEnd){
                Shift[] newshifts = Arrays.copyOf(shifts, shifts.length + 1);
                shifts = newshifts;
                shifts[i] = new Shift(i, 0, 0, 0);
                shifts[i].shiftId = i;
                shifts[i].shiftActivity = j;
                shifts[i].shiftStart = currentTime;
                shifts[i].shiftEnd = currentTime + activities[j].actDuration;
                currentTime += shiftIncrement;
                i += 1;
            }
        }
        nbShifts = i;
    }

    /**
     * Checks if there is at least a one-hour break between two shifts
     * @param shift1 First shift
     * @param shift2 Second shift
     * @return true if there is not enough break, false otherwise
     */
    boolean notEnoughBreak(Shift s1, Shift s2){
        return !(s1.shiftStart >= s2.shiftEnd + 3600 || s1.shiftEnd + 3600 <= s2.shiftStart);
    }

    /**
     * Checks if two shifts are compatible
     * @param shift1 First shift
     * @param shift2 Second shift
     * @return true if shifts overlap, false otherwise
     */    
    boolean overlappingShifts(Shift s1, Shift s2){
        return !(s1.shiftStart >= s2.shiftEnd || s1.shiftEnd <= s2.shiftStart);
    }

    /**
     * Checks if two time intervals overlap
     * @param interval1Start Start of first interval
     * @param interval1End End of first interval
     * @param interval2Start Start of second interval
     * @param interval2End End of second interval
     * @return true if intervals overlap, false otherwise
     */
    boolean overlappingIntervals(int interval1Start, int interval1End, int interval2Start, int interval2End){
        return !(interval1Start >= interval2End || interval1End <= interval2Start);
    }

    /**
     * Validates the instance data for consistency
     */
    void validateInstance() {
        // Check if all activities can be completed within their time windows
        for (Activity activity : activities) {
            if (activity.actDuration > (activity.actMaxEnd - activity.actMinStart)) {
                throw new IllegalArgumentException("Activity " + activity.activityId + " duration (" + (activity.actDuration / 3600) + "h) exceeds its time window");
            }
        }

        // Check if workers' availability windows are valid
        for (Agent agent : agents) {
            if (agent.availabilityStart >= agent.availabilityEnd) {
                throw new IllegalArgumentException("Agent " + agent.agentId + " has invalid availability window");
            }
        }

        // Check if activities can be performed within workers' availability

        Activity earliestStartActivity = Arrays.stream(activities).min(Comparator.comparingInt(a -> a.actMinStart)).orElseThrow();
        Activity latestEndActivity = Arrays.stream(activities).max(Comparator.comparingInt(a -> a.actMaxEnd)).orElseThrow();

        int earliestStart = earliestStartActivity.actMinStart;
        int latestEnd = latestEndActivity.actMaxEnd;
        
        for (Agent agent : agents) {
            if (agent.availabilityStart > earliestStart || agent.availabilityEnd < latestEnd) {
                System.out.println("Warning: Agent " + agent.agentId + " availability might not cover all tasks");
            }
        }
    }

    public static void main(String[] args) {
        if (args.length < 1) {
            System.err.println("Usage: java WorkforceSchedulingShifts inputFile [outputFile] [timeLimit]");
            System.exit(1);
        }
        try (HexalyOptimizer optimizer = new HexalyOptimizer()) {
            String instanceFile = args[0];
            String outputFile = args.length > 1 ? args[1] : null;
            String strTimeLimit = args.length > 2 ? args[2] : "20";

            WorkforceSchedulingShifts model = new WorkforceSchedulingShifts(optimizer);
            model.readInstance(instanceFile);
            model.solve(Integer.parseInt(strTimeLimit));
            
        } catch (Exception ex) {
            System.err.println(ex);
            ex.printStackTrace();
            System.exit(1);
        }
    }
}
