#include "optimizer/hexalyoptimizer.h"
#include <algorithm>
#include <fstream>
#include <iostream>
#include <set>
#include <string.h>
#include <vector>

using namespace hexaly;
using namespace std;

/**
 * Activity class represents a work item with its scheduling constraints
 * - id: Unique identifier for the activity
 * - minStart: minimum possible start time (in seconds)
 * - maxEnd: maximum possible end time (in seconds)
 * - duration: Duration of the activity (in seconds)
 */
class Activity {
  public:
    int id;
    int minStart;
    int maxEnd;
    int duration;

    Activity(int ID, int minS, int max, int d) {
        id = ID;
        minStart = minS;
        maxEnd = maxEnd;
        duration = d;
    }
};

/**
 * Agent class represents an employee with their availability window
 * - id: Unique identifier for the agent
 * - availabilityStart: Start of availability period (in seconds)
 * - availabilityEnd: End of availability period (in seconds)
 */
class Agent {
  public:
    int id;                // int
    int availabilityStart; // int
    int availabilityEnd;   // int

    Agent(int ID, int avStart, int avEnd) {
        id = ID;
        availabilityStart = avStart;
        availabilityEnd = avEnd;
    }
};

/**
 * Shift class represents a scheduled work period for an activity
 * - id: Unique identifier for the shift
 * - activityId: ID of the activity performed
 * - shiftStart: Start time of the shift (in seconds)
 * - shiftEnd: End time of the shift (in seconds)
 */
class Shift {
  public:
    int id;         // int
    int activityId; // int
    int shiftStart; // int
    int shiftEnd;   // int

    Shift(int ID, int actId, int shiftS, int shiftE) {
        id = ID;
        activityId = actId;
        shiftStart = shiftS;
        shiftEnd = shiftE;
    }
};

/**
 * Generates all possible shifts for activities with given time increment
 * @param activities Array of Activity objects
 * @param shiftIncrement Time increment in seconds
 * @return Array of generated Shift objects
 */
void generateShifts(vector<Shift> &shifts, vector<Activity> activities, int shiftIncrement) {
    int i = 0;
    for (int j = 0; j < activities.size(); j++) {
        int currentTime = activities[j].minStart;
        while (currentTime + activities[j].duration <= activities[j].maxEnd) {
            Shift s = Shift(i, activities[j].id, currentTime, currentTime + activities[j].duration);
            shifts.push_back(s);
            currentTime += shiftIncrement;
            i += 1;
        }
    }
};

/**
 * Checks if two  shifts are compatible
 * @param shif1 First shift
 * @param shift2 Second shift
 * @return true if shifts overlap, false otherwise
 */
bool incompatibleShifts(Shift shift1, Shift shift2) {
    int startShift1 = shift1.shiftStart;
    int endShift1 = shift1.shiftEnd;
    int startShift2 = shift2.shiftStart;
    int endShift2 = shift2.shiftEnd;
    int actId1 = shift1.activityId;
    int actId2 = shift2.activityId;
    return !(startShift1 >= endShift2 || endShift1 <= startShift2);
};

/**
 * Checks if there is at least a one-hour break between two shifts
 * @param shift1 First shift
 * @param shift2 Second shift
 * @return true if there is not enough break, false otherwise
 */
bool notEnoughBreak(Shift shift1, Shift shift2) {
    int startShift1 = shift1.shiftStart;
    int endShift1 = shift1.shiftEnd;
    int startShift2 = shift2.shiftStart;
    int endShift2 = shift2.shiftEnd;
    int actId1 = shift1.activityId;
    int actId2 = shift2.activityId;
    return !(startShift1 >= endShift2 + 3600 || endShift1 + 3600 <= startShift2);
};

/**
 * Checks if two time intervals overlap
 * @param interval1Start Start of first interval
 * @param interval1End End of first interval
 * @param interval2Start Start of second interval
 * @param interval2End End of second interval
 * @return true if intervals overlap, false otherwise
 */
bool overlappingIntervals(int int1Start, int int1End, int int2Start, int int2End) {
    return !(int1Start >= int2End || int1End <= int2Start);
};

class WorkshopSchedulingShifts {
  private:
    // Number of agents
    int nbAgents;
    // Number of activities
    int nbActivities;
    // Number of days
    int timeHorizon;
    // Increment
    int shiftIncrement;

    // Activities informations
    vector<Activity> activities;
    // Agents informations
    vector<Agent> agents;
    // Possible shifts
    vector<Shift> shifts;

    // Hexaly Optimizer
    HexalyOptimizer optimizer;
    // Decision variables: time range of each task
    vector<vector<HxExpression>> agentShift;
    // Objective = minimize the missing agent time and total working time of the agents
    HxExpression totalUnderstaffing;
    HxExpression totalWorkingTime;

  public:
    // The input files follow the "Taillard" format
    void readInstance(const string &fileName) {
        ifstream infile;
        if (!infile) {
            cerr << "Unable to open file: " << fileName << endl;
        }
        infile.open(fileName.c_str());

        std::string line;
        std::vector<std::string> elements;

        while (std::getline(infile, line)) {
            std::istringstream iss(line);
            std::string elem;
            while (iss >> elem) {
                if (elem[0] == '#')
                    break; // Ignore the rest of the line if a comment is found
                elements.push_back(elem);
            }
        }

        // Read problem dimensions
        auto it = elements.begin();
        nbAgents = std::stoi(*it++);
        nbActivities = std::stoi(*it++);
        timeHorizon = std::stoi(*it++);
        shiftIncrement = std::stof(*it++) * 3600;
        infile.ignore(numeric_limits<streamsize>::max(), '\n');

        // Read activities informations
        // Duration time of each activity
        for (int i = 0; i < nbActivities; i++) {
            int activityDuration;
            activityDuration = stoi(*it++);
            activities.push_back(Activity(i, 0, 0, activityDuration * 3600));
        }
        // Maximum end and minimum possible start hour of each activity
        for (int i = 0; i < nbActivities; i++) {
            int minStart = std::stoi(*it++);
            int maxEnd = std::stoi(*it++);
            activities[i].minStart = minStart * 3600;
            activities[i].maxEnd = maxEnd * 3600;
        }

        // Read agents informations
        for (int i = 0; i < nbAgents; i++) {
            int availabilityStart = std::stoi(*it++);
            int availabilityEnd = std::stoi(*it++);
            infile >> availabilityStart >> availabilityEnd;
            agents.push_back(Agent(i, availabilityStart * 3600, availabilityEnd * 3600));
        }
        // Generation of all possible shifts for each activity with a given increment
        generateShifts(shifts, activities, shiftIncrement);

        // Validates the instance data
        validateInstance();
        infile.close();
    }

    /**
     * Validates the instance data for consistency
     */
    void validateInstance() {
        // Check if all activities can be completed within their time windows
        for (Activity activity : activities) {
            if (activity.duration > (activity.maxEnd - activity.minStart)) {
                throw("Activity " + to_string(activity.id) + " duration (" + to_string(activity.duration / 3600) +
                      "h) exceeds its time window");
            }
        }

        // Check if workers' availability windows are valid
        for (Agent agent : agents) {
            if (agent.availabilityStart >= agent.availabilityEnd) {
                throw("Agent " + to_string(agent.id) + " has invalid availability window");
            }
        }

        // Check if activities can be performed within workers' availability
        auto earliestStartIt =
            std::min_element(activities.begin(), activities.end(),
                             [](const Activity &a, const Activity &b) { return a.minStart < b.minStart; });
        auto latestEndIt =
            std::max_element(activities.begin(), activities.end(),
                             [](const Activity &a, const Activity &b) { return a.minStart < b.minStart; });
        int earliestStart = earliestStartIt->minStart;
        int latestEnd = latestEndIt->maxEnd;

        for (Agent agent : agents) {
            if (agent.availabilityStart > earliestStart || agent.availabilityEnd < latestEnd) {
                cout << "Warning: Agent " << agent.id << " availability might not cover all tasks" << endl;
            }
        }
    }

    void solve(int timeLimit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        // Decision variables : AgentShift[a][s] is true when the agent a works the shift s
        agentShift.resize(nbAgents);
        for (unsigned int i = 0; i < nbAgents; i++) {
            agentShift[i].resize(shifts.size());
            for (unsigned int j = 0; j < shifts.size(); j++) {
                agentShift[i][j] = model.boolVar();
            }
        }
        // Constraints
        // 1. The shifts performed by an agent must not overlap
        for (int a = 0; a < nbAgents; a++) {
            for (auto s1 = shifts.begin(); s1 != shifts.end(); s1++) {
                for (auto s2 = shifts.begin(); s2 != shifts.end(); s2++) {
                    if (incompatibleShifts(*s1, *s2) && s1 != s2) {
                        model.constraint(agentShift[a][s1->id] + agentShift[a][s2->id] <= 1);
                    }
                }
            }
        }

        // 2. An agent must take at least a one-hour break between two shifts
        for (int a = 0; a < nbAgents; a++) {
            for (auto s1 = shifts.begin(); s1 != shifts.end(); s1++) {
                for (auto s2 = shifts.begin(); s2 != shifts.end(); s2++) {
                    if (notEnoughBreak(*s1, *s2) && s1 != s2) {
                        model.constraint(agentShift[a][s1->id] + agentShift[a][s2->id] <= 1);
                    }
                }
            }
        }

        // 3. Each agent must work exactly two shifts in the day
        for (int a = 0; a < nbAgents; a++) {
            HxExpression shiftsDone = model.sum();
            for (int id = 0; id < shifts.size(); id++) {
                shiftsDone.addOperand(agentShift[a][id]);
            }
            model.constraint(shiftsDone == 2);
        }

        // Objective 1 : Minimize understaffing
        HxExpression totalUnderstaffing = model.sum();
        for (auto activity = activities.begin(); activity != activities.end(); activity++) {
            int currentTime = activity->minStart;
            while (currentTime < activity->maxEnd) {
                // Current interval of time
                int currentIntervalStart = currentTime;
                int currentIntervalEnd = currentTime + shiftIncrement;
                // The activity will not be pursued after its maximum possible end
                if (currentIntervalEnd > activity->maxEnd) {
                    currentIntervalEnd = activity->maxEnd;
                }
                // Shifts performed in the current time interval
                vector<Shift> shiftsAtCurr;
                for (auto shift = shifts.begin(); shift != shifts.end(); shift++) {
                    if (shift->activityId != activity->id) {
                        continue;
                    }
                    if (overlappingIntervals(currentIntervalStart, currentIntervalEnd, shift->shiftStart,
                                             shift->shiftEnd)) {
                        shiftsAtCurr.push_back(*shift);
                    }
                }
                // Number of agents currently performing the task
                HxExpression totalAgentsWorking = model.sum();
                for (auto shift = shiftsAtCurr.begin(); shift != shiftsAtCurr.end(); shift++) {
                    for (int a = 0; a < nbAgents; a++) {
                        totalAgentsWorking.addOperand(agentShift[a][shift->id]);
                    }
                }
                // Number of agents missing
                totalUnderstaffing.addOperand(shiftIncrement * model.max(0, 1 - totalAgentsWorking));
                currentTime += shiftIncrement;
            }
        }
        model.minimize(totalUnderstaffing);

        // Objective 2 : Minimize the total worktime of all the agents
        HxExpression totalWorkingTime = model.sum();
        for (int a = 0; a < nbAgents; a++) {
            for (auto shift = shifts.begin(); shift != shifts.end(); shift++) {
                totalWorkingTime.addOperand(agentShift[a][shift->id] * activities[shift->activityId].duration);
            }
        }
        model.minimize(totalWorkingTime);
        model.close();

        // Parametrize the optimizer
        optimizer.getParam().setTimeLimit(10);
        optimizer.solve();
    }
};

int main(int argc, char **argv) {
    if (argc < 2) {
        cerr << "Usage: workforce_scheduling_shifts inputFile [outputFile] [timeLimit]" << endl;
        return 1;
    }
    const char *instanceFile = argv[1];
    const char *strTimeLimit = argc > 2 ? argv[2] : "5";
    try {
        WorkshopSchedulingShifts model;
        model.readInstance(instanceFile);
        model.solve(atoi(strTimeLimit));
    } catch (const exception &e) {
        cerr << "An error occurred: " << e.what() << endl;
        return 1;
    }
};
