import hexaly.optimizer
import sys

# Activity class represents a work item with its scheduling constraints
#   - id: Unique identifier for the task
#   - min_start: minimum possible start time (in seconds)
#   - max_end: maximum possible end time (in seconds)
#   - duration: Duration of the task (in seconds)
class Activity:
    def __init__(self, id, min_start, max_end, duration):
        self.id = id #int
        self.min_start = min_start #int
        self.max_end = max_end #int
        self.duration = duration #int

# Agent class represents an employee with their availability window
#   - id: Unique identifier for the agent
#   - availability_start: Start of availability period (in seconds)
#   - availability_end: End of availability period (in seconds)
class Agent:
    def __init__(self, id, av_start, av_end):
        self.id = id #int
        self.availability_start = av_start #int
        self.availability_end = av_end #int

# Shift class represents a work item with its scheduling constraints
#   - id: Unique identifier for the shift
#   - activity_id: ID of the activity performed
#   - shift_start: Start time of the shift (in seconds)
#   - shift_end: End time of the shift (in seconds)
class Shift:
    def __init__(self, id, activity_id, shift_start, shift_end):
        self.id = id #int
        self.activity_id = activity_id #int
        self.shift_start = shift_start #int
        self.shift_end = shift_end #int


#  Generates all possible shifts for activities with given time increment
#  @param activities Array of Activity objects
#  @param shift_increment Time increment in seconds
#  @return Array of generated Shift objects
def generateShifts(activities, shift_increment):
    i = 0
    shifts = []
    for a in activities:
        current_time = a.min_start
        while current_time + a.duration <= a.max_end:
            shifts.append(Shift(i, a.id, current_time, current_time + a.duration))
            i += 1
            current_time += shift_increment
    return shifts


# Checks if two shifts are compatible
#   @param slot1 First shift
#   @param slot2 Second shift
#   @return true if shifts overlap, false otherwise
def incompatibleShifts(shift1, shift2):
    startShift1 = shift1.shift_start
    endShift1 = shift1.shift_end
    startShift2 = shift2.shift_start
    endShift2 = shift2.shift_end
    return not (startShift1 >= endShift2 or endShift1 <= startShift2)


# Checks if there is at least a one-hour break between two shifts
#   @param slot1 First shift
#   @param slot2 Second shift
#   @return true if there is not enough break, false otherwise
def notEnoughBreak(shift1, shift2):
    startShift1 = shift1.shift_start
    endShift1 = shift1.shift_end
    startShift2 = shift2.shift_start
    endShift2 = shift2.shift_end
    return not (startShift1 >= endShift2 + 3600 or endShift1 + 3600 <= startShift2)


# Checks if two time intervals overlap
#   @param interval1Start Start of first interval
#   @param interval1End End of first interval
#   @param interval2Start Start of second interval
#   @param interval2End End of second interval
#   @return true if intervals overlap, false otherwise
def overlappingIntervals(int1_start, int1_end, int2_start, int2_end):
    return not (int1_start >= int2_end or int1_end <= int2_start)


if len(sys.argv) < 2:
    print("Usage: python tsp.py inputFile [outputFile] [timeLimit]")
    sys.exit(1)

# Filters out comments from a text file
#   @param filename Path to input file
def read_elem(filename):
    with open(filename) as f:
        lines = f.readlines()
        result = []
        for line in lines:
            line = line.split("#")[
                0
            ].strip()  # Split at '#' and take the part before it
            if line:  # Only add non-empty lines
                result.extend(
                    line.split()
                )  # Split the line into elements and add them to the result
    return result


# Validates the instance data for consistency
def validateInstance():
    # Check if all activities can be completed within their time windows
    for activity in activities:
        if activity.duration > (activity.max_end - activity.min_start):
            raise ValueError(
                "Activity "
                + activity.id
                + " duration ("
                + (activity.duration / 3600)
                + "h) exceeds its time window"
            )

    # Check if workers' availability windows are valid
    for agent in agents:
        if agent.availability_start >= agent.availability_end:
            raise ValueError("Agent " + agent.id + " has invalid availability window")

    # Check if activities can be performed within workers' availability
    earliest_start = min([act.min_start for act in activities])
    latest_end = max([act.max_end for act in activities])

    for agent in agents:
        if (
            agent.availability_start > earliest_start
            or agent.availability_end < latest_end
        ):
            print(
                "Warning: Agent "
                + str(agent.id)
                + " availability might not cover all tasks"
            )


with hexaly.optimizer.HexalyOptimizer() as optimizer:
    #
    # Read path of input file
    #
    file_it = iter(read_elem(sys.argv[1]))

    # Read dimensions of the problem
    nb_agents = int(next(file_it))
    nb_activities = int(next(file_it))
    time_horizon = int(next(file_it))
    shift_increment = round(float(next(file_it)) * 60)

    # Read activities informations
    # Duration time of each activity
    activities = []
    for i in range(nb_activities):
        activities.append(Activity(i, 0, 0, int(next(file_it)) * 3600))
    # Maximum end and minimum possible start hour of each activity
    for i in range(nb_activities):
        activities[i].min_start = int(next(file_it)) * 3600
        activities[i].max_end = int(next(file_it)) * 3600

    # Read agents informations
    # Availability period in the day of each agent
    agents = [
        Agent(i, int(next(file_it)), int(next(file_it))) for i in range(nb_agents)
    ]

    # Generation of all possible shifts for each activity with a given increment
    shifts = generateShifts(activities, shift_increment)

    # Validates the instance data
    validateInstance()

    #
    # Declare the optimization model
    #
    model = optimizer.model

    # Decision variables : agent_shift[a][s] is true when the agent a works the shift s
    agent_shift = [[model.bool() for j in range(len(shifts))] for i in range(nb_agents)]

    # Constraints
    # 1. The shifts performed by an agent must not overlap
    for a in range(nb_agents):
        for s1 in shifts:
            for s2 in shifts:
                if incompatibleShifts(s1, s2) and s1 != s2:
                    model.constraint(agent_shift[a][s1.id] + agent_shift[a][s2.id] <= 1)

    # 2. An agent must take at least a one-hour break between two shifts
    for a in range(nb_agents):
        for s1 in shifts:
            for s2 in shifts:
                if notEnoughBreak(s1, s2) and s1 != s2:
                    model.constraint(agent_shift[a][s1.id] + agent_shift[a][s2.id] <= 1)

    # 3. Each agent must work exactly two shifts in the day
    for a in range(nb_agents):
        model.constraint(sum([agent_shift[a][s.id] for s in shifts]) == 2)

    # Objective 1 :  Minimize understaffing
    # For each activity and at each interval of time of the increment length
    total_understaffing = 0
    for activity in activities:
        current_time = activity.min_start
        while current_time < activity.max_end:
            # Current interval of time
            current_interval_start = current_time
            current_interval_end = current_time + shift_increment
            # The activity will not be pursued after its maximum possible end
            if current_interval_end > activity.max_end:
                current_interval_end = activity.max_end
            # Shifts performed in the current time interval
            shifts_at_current = []
            for shift in shifts:
                if shift.activity_id != activity.id:
                    continue
                if overlappingIntervals(
                    current_interval_start,
                    current_interval_end,
                    shift.shift_start,
                    shift.shift_end,
                ):
                    shifts_at_current.append(shift)
            # Number of agents currently performing the task
            total_agents_working = 0
            for shift in shifts_at_current:
                agents_working = sum(
                    [agent_shift[a][shift.id] for a in range(nb_agents)]
                )
                total_agents_working += agents_working
            # Number of agents missing
            total_understaffing += model.max(0, 1 - total_agents_working)
            current_time += shift_increment
    total_understaffing *= shift_increment
    model.minimize(total_understaffing)

    # Objective 2 : Minimize the total worktime of all the agents
    total_working_time = 0
    for a in range(nb_agents):
        total_working_time += sum(
            [agent_shift[a][s.id] * activities[s.activity_id].duration for s in shifts]
        )
    model.minimize(total_working_time)
    model.close()

    # Parameterize the optimizer
    if len(sys.argv) >= 4:
        optimizer.param.time_limit = int(sys.argv[3])
        optimizer.param.verbosity = 2
    else:
        optimizer.param.time_limit = 5
    optimizer.solve()
